package com.atlassian.diagnostics.ipd.api.registry;

import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.MeterTag;
import com.atlassian.diagnostics.ipd.api.meters.CounterMeter;
import com.atlassian.diagnostics.ipd.api.meters.CustomMeter;
import com.atlassian.diagnostics.ipd.api.meters.CustomMeter.MBeanSupplier;
import com.atlassian.diagnostics.ipd.api.meters.IpdMeter;
import com.atlassian.diagnostics.ipd.api.meters.JmxCopyMeter;
import com.atlassian.diagnostics.ipd.api.meters.StatsMeter;
import com.atlassian.diagnostics.ipd.api.meters.ValueAndStatsMeterWrapper;
import com.atlassian.diagnostics.ipd.api.meters.ValueMeter;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterFactory;
import com.atlassian.diagnostics.ipd.api.meters.custom.type.IpdConnectionState;

import javax.annotation.Nullable;
import javax.management.ObjectName;
import java.util.Collection;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

/**
 * Interface for managing and registering various types of meters in the IPD (In-Product Diagnostics) registry.
 * All meters are registered with a unique key, which consists of a name and a set of tags.
 * By default, meters are visible through JMX and log files.
 * <p>
 * Provides methods to register, retrieve, and manage meters such as counters, custom meters, value meters, and stats meters.
 * </p>
 *
 * @since 3.0.0
 */
public interface IpdRegistry {


    /**
     * returns an `IpdMeter` of type 'T' with the given name and tags.
     * If a meter with the given name and tags does not exist, a new one will be created using the provided factory.
     *
     * @param factory the factory to use to create the meter instance
     * @param name the name of the meter
     * @param tags the tags of the meter
     * @param <T> the type of the meter
     * @return existing or the newly registered `IpdMeter`
     */
    <T extends IpdMeter> T register(MeterFactory<T> factory, String name, MeterTag... tags);

    /**
     * Retrieves an `IpdMeter` by its `MeterKey`.
     *
     * @param meterKey the key of the meter to retrieve
     * @return the `IpdMeter` associated with the given key, or `null` if no such meter exists
     */
    @Nullable
    IpdMeter get(MeterKey meterKey);

    /**
     * Retrieves all registered `IpdMeter` instances as an immutable set.
     *
     * @return an immutable set of all registered `IpdMeter` instances
     */
    Collection<IpdMeter> getMeters();

    /**
     * Retrieves all registered `IpdMeter` instances with the given prefix as an immutable set.
     *
     * @param prefix the prefix to filter by
     * @return an immutable set of all registered `IpdMeter` instances with the given prefix
     */
    Collection<IpdMeter> getMeters(String prefix);

    /**
     * Retrieves all keys of registered `IpdMeter` instances as an immutable set.
     *
     * @return an immutable set of all registered meter keys
     */
    Set<MeterKey> getMeterKeys();

    /**
     * Retrieves all keys of registered `IpdMeter` instances with the given prefix as an immutable set.
     *
     * @param prefix the prefix to filter by
     * @return an immutable set of all registered meter keys with the given prefix
     */
    Set<MeterKey> getMeterKeys(String prefix);

    /**
     * Unregisters all disabled metrics.
     * <p>
     * This method iterates through all registered `IpdMeter` instances and unregisters from JMX those that are disabled.
     * Meters are considered disabled if their `IpdMeter#isEnabled` method returns `false`.
     * </p>
     */
    void unregisterAllDisabledMetrics();

    /**
     * Removes the `IpdMeter` associated with the given `MeterKey`. Does nothing if no such meter exists.
     * <p>
     * The existing instance of IpdMeter will be closed. See {@link IpdMeter#close()}.
     * </p>
     * @param meterKey the key of the meter to remove
     */
    void remove(MeterKey meterKey);

    /**
     * Removes all registered `IpdMeter` instances.
     */
    void removeAll();

    /**
     * Removed all registered `IpdMeter` instances with the given prefix and tags.
     *
     * <p>
     *     If any of the tags are not present in the meter, it will not be removed, but
     *     any additional tags in the meter will not prevent it from being removed.
     * </p>
     *
     * @param prefix the prefix to filter by
     * @param tags the tags to filter by
     */
    void removeAll(String prefix, MeterTag... tags);

    /**
     * Removes all registered `IpdMeter` instances that match prefix and any of the given tags.
     *
     * <p>
     *     If none of the tags are present in the meter, it will not be removed.
     * </p>
     * <p>
     *     Useful for removing a few meters with different tags, but the same prefix.
     *     For example, removing user directory metrics:
     *     <pre>
     *         removeAllWithAnyTag("user.directory",
     *              Set.of(
     *                  MeterTag.of("name", "ldap"),
     *                  MeterTag.of("name", "crowd")));
     *     </pre>
     *     This will remove all meters with prefix "user.directory" and tags "name=ldap" or "name=crowd", but not "name=my_directory".
     *
     * @param prefix the prefix to filter by
     * @param tags the tags to filter by
     */
    void removeAllWithAnyTag(String prefix, Set<MeterTag> tags);

    /**
     * Removes all registered `IpdMeter` instances that match the given predicate.
     *
     * @param predicate the predicate to filter by
     */
    void removeIf(Predicate<IpdMeter> predicate);

    /**
     * Removes all registered `IpdMeter` instances with the given prefix that also match the given predicate.
     *
     * @param prefix the prefix to filter by
     * @param predicate the predicate to filter by
     */
    void removeIf(String prefix, Predicate<IpdMeter> predicate);

    /**
     * Iterates through all registered metrics matching prefix and removes those that do not match the predicate.
     *
     * For example, removing user directory metrics:
     * <pre>
     *     retainIf("user.directory",
     *         meter -> "ldap".equals(meter.getKey().getTag("name")));
     * </pre>
     * After this call, meters that match "user.directory" but don't have tag "name=ldap" will be removed.
     *
     * @param prefix the prefix to filter by
     * @param predicate the predicate to filter by
     */
    void retainIf(String prefix, Predicate<IpdMeter> predicate);

    /**
     * Returns metric wrapper containing both value and statistic metrics.
     *
     * @param name meter name
     * @param tags meter tags
     * @return Wrapper containing existing or new meters
     */
    default ValueAndStatsMeterWrapper valueAndStats(String name, MeterTag... tags) {
        return new ValueAndStatsMeterWrapper(stats(name, tags), value(name, tags));
    }

    /**
     * Returns value meter for a given name and tags.
     *
     * @param name meter name
     * @param tags meter tags
     * @return Existing or new meter
     */
    ValueMeter value(String name, MeterTag... tags);

    /**
     * Sets value to the existing or new value meter.
     * Equivalent to <pre>ipdRegistry.value(name, tags).update(value)</pre>
     *
     * @param name meter name
     * @param value value to set
     * @param tags meter tags
     */
    default void setValue(String name, long value, MeterTag...  tags) {
        value(name, tags).update(value);
    }
    /**
     * Returns stats meter for a given name and tags.
     *
     * @param name meter name
     * @param tags meter tags
     * @return Existing or new meter
     */
    StatsMeter stats(String name, MeterTag... tags);

    /**
     * Updates stats meter with a given value.
     * Equivalent to <pre>ipdRegistry.stats(name, tags).update(value)</pre>
     * @param name meter name
     * @param value value to set
     * @param tags meter tags
     */
    default void addStats(String name, long value, MeterTag... tags) {
        stats(name, tags).update(value);
    }

    /**
     * Updates stats meter with a given value.
     * Equivalent to <pre>ipdRegistry.stats(name, tags).update(value, timeUnit)</pre>
     * @param name meter name
     * @param value value to set
     * @param timeUnit time unit of the value
     * @param tags meter tags
     */
    default void addStats(String name, long value, TimeUnit timeUnit, MeterTag... tags) {
        stats(name, tags).update(value, timeUnit);
    }

    /**
     * Returns counter meter for a given name and tags.
     *
     * @param name meter name
     * @param tags meter tags
     * @return Existing or new meter
     */
    CounterMeter counter(String name, MeterTag... tags);

    /**
     * Increments counter meter by 1.
     * Equivalent to <pre>ipdRegistry.counter(name, tags).increment(1)</pre>
     *
     * @param name meter name
     * @param tags meter tags
     */
    default void incrementCounter(String name, MeterTag... tags) {
        counter(name, tags).increment(1);
    }

    /**
     * Returns custom meter for a given name, tags and type.
     * <p>
     * 'type' parameter points to class which implements interface annotated with <b>@MXBean</b>.
     * Getters in this interface will define meter attributes.<br/>
     * This class has to have no argument constructor.
     * </p>
     * See 'type' example implementation {@link IpdConnectionState}
     *
     * @param name metric name
     * @param type points to custom meter MBean implementation. This class has to implement interface annotated with <b>@MXBean</b>. Getters in this interface will define meter attributes.
     * @param tags meter tags
     * @param <T> Type defining meter attributes
     * @return existing or new meter
     */
    <T> CustomMeter<T> custom(String name, Class<T> type, MeterTag... tags);

    /**
     * Returns custom meter for a given name, tags and instance.
     * <p>
     * 'instance' object has to be of class which implements interface annotated with <b>@MXBean</b>.
     * Getters in this interface will define meter attributes.<br/>
     * </p>
     * See 'type' example implementation {@link IpdConnectionState}
     *
     * <p>
     *     <b>Important note:</b><br/>
     *     Returned meter will reference given instance object only on creation.
     *     If CustomMeter for a give name and tags is already registered,
     *     it will be returned with the existing instance object and provided 'instance' parameter will be ignored.
     * </p>
     *
     * @param name metric name
     * @param instance this objects class has to implement interface annotated with <b>@MXBean</b>. Getters in this interface will define meter attributes.
     * @param tags meter tags
     * @param <T> Type defining meter attributes
     * @return existing or new meter
     */
    <T> CustomMeter<T> custom(String name, MBeanSupplier<T> instance, MeterTag... tags);

    /**
     * Returns an IPD meter which copies stats attributes of existing JMX meter.
     * All attribute changes in the original metric will be reflected in JMX and log file after calling {@link JmxCopyMeter#update()}.
     *
     * @param name meter name with postfix
     * @param objectToCopy object name of the JMX bean to copy attributes from
     * @param tags meter tags
     * @return existing or new meter
     */
    JmxCopyMeter statsCopy(String name, ObjectName objectToCopy, MeterTag... tags);


    /**
     * Shuts down the registry and unregisters all meters.
     */
    void shutdown();

}
