package com.atlassian.diagnostics.ipd.api.registry;

import com.atlassian.diagnostics.ipd.api.MeterConfigurations;
import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.MeterTag;
import com.atlassian.diagnostics.ipd.api.meters.IpdMeter;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfigBuilderImpl;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterFactory;
import com.atlassian.diagnostics.ipd.api.meters.config.ProductMeterConfigBuilder;
import com.google.common.base.Strings;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static java.util.Collections.disjoint;
import static java.util.Collections.unmodifiableCollection;
import static java.util.Collections.unmodifiableSet;

/**
 * Abstract implementation of {@link IpdRegistry} that provides common functionality of managing IPD meters.
 *
 * @since 3.0.0
 */
public abstract class AbstractIpdMeterRegistry implements IpdRegistry {
    private static final BiConsumer<MeterKey, ProductMeterConfigBuilder> EMPTY_CONSUMER = (ignored1, ignored2) -> {};
    protected final MeterConfigurations meterConfigurations;
    protected final Map<MeterKey, IpdMeter> meters;

    public AbstractIpdMeterRegistry(MeterConfigurations meterConfigurations) {
        this(meterConfigurations, new ConcurrentHashMap<>());
    }

    public AbstractIpdMeterRegistry(final MeterConfigurations meterConfigurations,
                                    final Map<MeterKey, IpdMeter> meters) {
        this.meterConfigurations = meterConfigurations;
        this.meters = meters;
    }

    @Override
    public <T extends IpdMeter> T register(final MeterFactory<T> factory, final String name, final MeterTag... tags) {
        final MeterKey key = new MeterKey(mergeMeterName(name, factory.typeSuffix()), tags);
        final IpdMeter ipdMeter = meters.computeIfAbsent(key, k -> factory.createMetric(evaluateConfig(k)));
        if (ipdMeter.getTypeId().equals(factory.getTypeId())) {
            //noinspection unchecked
            return (T) ipdMeter;
        }
        throw new IpdRegisterException(String.format("Ipd meter type different than expected, key= %s, expectedType=%s, actualType=%s", ipdMeter.getMeterKey(), factory.getTypeId(), ipdMeter.getTypeId()));
    }

    private MeterConfig evaluateConfig(final MeterKey key) {
        final ProductMeterConfigBuilder productMeterConfigBuilder = new MeterConfigBuilderImpl();
        meterConfigurations.evaluateConfig(key, productMeterConfigBuilder);
        return productMeterConfigBuilder.build(key);
    }

    @Nullable
    @Override
    public IpdMeter get(final MeterKey meterKey) {
        return meters.get(meterKey);
    }

    @Override
    public void remove(final MeterKey meterKey) {
        final IpdMeter meter = meters.remove(meterKey);
        if (meter != null) {
            meter.close();
        }
    }

    @Override
    public void removeAll() {
        meters.values().forEach(IpdMeter::close);
        meters.clear();
    }

    @Override
    public void removeAll(final String prefix, final MeterTag... tags) {
        final var tagsSet = Set.of(tags);
        removeIf(prefix, meter -> tagsSet.containsAll(List.of(meter.getMeterKey().getTags())));
    }

    @Override
    public void removeAllWithAnyTag(final String prefix, final Set<MeterTag> tags) {
        removeIf(prefix, meter -> !disjoint(Set.of(meter.getMeterKey().getTags()), tags));
    }

    @Override
    public void removeIf(final Predicate<IpdMeter> predicate) {
        meters.values().stream()
                .filter(predicate)
                .forEach(IpdMeter::close);
        meters.values().removeIf(predicate);
    }

    @Override
    public void removeIf(final String prefix, final Predicate<IpdMeter> predicate) {
        removeIf(meter -> meter.getMeterKey().getName().startsWith(prefix) && predicate.test(meter));
    }

    @Override
    public void retainIf(final String prefix, final Predicate<IpdMeter> predicate) {
        removeIf(prefix, predicate.negate());
    }

    @Override
    public Collection<IpdMeter> getMeters() {
        return unmodifiableCollection(meters.values());
    }

    @Override
    public Collection<IpdMeter> getMeters(final String prefix) {
        return meters.values().stream()
                .filter(meter -> meter.getMeterKey().getName().startsWith(prefix))
                .toList();
    }

    @Override
    public Set<MeterKey> getMeterKeys() {
        return unmodifiableSet(meters.keySet());
    }

    @Override
    public Set<MeterKey> getMeterKeys(final String prefix) {
        return meters.keySet().stream()
                .filter(key -> key.getName().startsWith(prefix))
                .collect(Collectors.toUnmodifiableSet());
    }

    @Override
    public void unregisterAllDisabledMetrics() {
        getMeters().stream()
                .filter(ipdMetric -> !ipdMetric.isEnabled())
                .forEach(IpdMeter::hideUntilUpdate);
    }

    @Override
    public void shutdown() {
        removeAll();
    }

    private static String mergeMeterName(final String name, final String suffix) {
        if (Strings.isNullOrEmpty(suffix)) {
            return name;
        }
        return name + "." + suffix;
    }
}

