package com.atlassian.diagnostics.ipd.api.meters.config;

import com.atlassian.diagnostics.ipd.api.IpdConstants;
import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.MeterTag;
import com.atlassian.diagnostics.ipd.api.meters.IpdMeter;
import org.slf4j.Logger;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import java.time.Clock;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig.Properties.CUSTOM_CLOCK;
import static com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig.Properties.INTERVAL_LOGGING;
import static com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig.Properties.LOG_ON_UPDATE;
import static com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig.Properties.WORK_IN_PROGRESS;

/**
 * @since 5.0.0
 */
public class MeterConfigBuilderImpl implements ProductMeterConfigBuilder {
    final Map<String, Object> properties = new HashMap<>();
    Predicate<MeterConfig> enabledCheck = IpdConstants.TRUE_CONFIG_PREDICATE;
    Consumer<IpdMeter> updateListener = IpdConstants.NO_OP;
    MeterObjectNameConstructor objectNameFactory = MeterConfigBuilderImpl::defaultTestObjectNameConstructor;
    Predicate<IpdMeter> shouldLog = IpdConstants.TRUE_METER_PREDICATE;
    Logger logger = IpdConstants.DEFAULT_DATA_LOGGER;

    @Override
    public ProductMeterConfigBuilder setObjectNameConstructor(MeterObjectNameConstructor objectNameFactory) {
        this.objectNameFactory = objectNameFactory;
        return this;
    }

    @Override
    public MeterConfigBuilderImpl setUpdateListener(Consumer<IpdMeter> updateListener) {
        this.updateListener = updateListener;
        return this;
    }

    @Override
    public MeterConfigBuilder addEnabledCheck(Predicate<MeterConfig> enabledCheck) {
        this.enabledCheck = this.enabledCheck.and(enabledCheck);
        return this;
    }

    @Override
    public MeterConfigBuilder setProperty(String key, Object value) {
        properties.put(key, value);
        return this;
    }

    @Override
    public MeterConfigBuilder setWorkInProgress(boolean workInProgress) {
        setProperty(WORK_IN_PROGRESS, workInProgress);
        return this;
    }

    @Override
    public MeterConfigBuilder logOnceEveryInterval(final int interval) {
        final AtomicInteger counter = new AtomicInteger(0);
        return logWhen((m) -> counter.incrementAndGet() % interval == 0);
    }

    @Override
    public MeterConfigBuilder logIfUpdatedInLastMinute() {
        final var clock = Optional.ofNullable(properties.get(CUSTOM_CLOCK))
                .map(Clock.class::cast)
                .orElse(Clock.systemUTC());
        return logWhen((m) -> clock.millis() - m.lastUpdateMillis() < 60_000);
    }

    @Override
    public MeterConfigBuilder setCustomLogger(final Logger logger) {
        this.logger = logger;
        return this;
    }

    @Override
    public MeterConfigBuilder setIntervalLogging(final boolean enabled) {
        this.properties.put(INTERVAL_LOGGING, enabled);
        return this;
    }

    @Override
    public MeterConfigBuilder setLogOnUpdate(final boolean logOnUpdate) {
        this.properties.put(LOG_ON_UPDATE, logOnUpdate);
        return this;
    }

    @Override
    public MeterConfigBuilder logWhen(final Predicate<IpdMeter> shouldLog) {
        this.shouldLog = this.shouldLog.and(shouldLog);
        return this;
    }

    public MeterConfig build(MeterKey meterKey) {
        return new MeterConfig(meterKey, this);
    }

    private static ObjectName defaultTestObjectNameConstructor(final MeterKey meterKey) {
        var tags = Arrays.stream(meterKey.getTags()).map(MeterTag::toString).collect(Collectors.joining(","));
        var name = "com.atlassian.diagnostics:meterKeyProperty=" + meterKey.getName() + (tags.isEmpty() ? "" : "," + tags);

        try {
            return new ObjectName(name);
        } catch (final MalformedObjectNameException e) {
            throw new RuntimeException("ObjectName: " + name, e);
        }
    }

}
