package com.atlassian.diagnostics.ipd.api.meters.config;

import com.atlassian.diagnostics.ipd.api.meters.IpdMeter;
import org.slf4j.Logger;

import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * Builder used to dynamically create a {@link MeterConfig}.
 * @since 5.0.0
 */
public interface MeterConfigBuilder {
    /**
     * Adds a check to the MeterConfigBuilder that will be used to determine if the MeterConfig is enabled.
     * <p>
     *     Adding a new check will not remove the previous checks.
     *     All checks must pass for the MeterConfig to be enabled.
     * </p>
     * @param enabledCheck predicate returning true if the Meter should be enabled
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder addEnabledCheck(Predicate<MeterConfig> enabledCheck);

    /**
     * Sets meter as "work in progress". Work in progress meters require additional feature flag to be enabled.
     * It is intended for meter we want to test on internal instances before enabling them on production.
     * @param workInProgress true if the meter is work in progress
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder setWorkInProgress(boolean workInProgress);

    /**
     * Sets a property on the MeterConfigBuilder.
     * Properties can be accessed later on for individual meters through {@link MeterConfig#getProperties()}.
     *
     * @param key property key
     * @param value property value
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder setProperty(String key, Object value);

    /**
     * Makes the meter to be logged once every 'interval' times.
     * <p>
     *     By default, the 'interval' is set to 1, which usually means the meter will be logged every minute.
     * </p>
     * <p>
     *     If product logs meters every minute, setting the 'interval' value to 5 will make the meter to be logged every 5 minutes.
     * </p>
     *
     * @param interval how often the meter should be logged. Is a positive integer.
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder logOnceEveryInterval(int interval);

    /**
     * Makes the meter to be logged only if it was updated in the last minute.
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder logIfUpdatedInLastMinute();

    /**
     * Sets a custom logger for the meter. This logger will be used to log the meter values.
     * <p>
     *     By default, the meter is logged to ipd log file using default logger defined in
     * {@link com.atlassian.diagnostics.ipd.api.IpdConstants#DEFAULT_DATA_LOGGER}.
     * </p>
     *
     * @param logger custom logger
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder setCustomLogger(Logger logger);

    /**
     * Sets the meter to log at regular intervals.
     * <p>
     *     By default, the meters are automatically logged in a regular interval.
     * </p>
     * Setting the 'enabled' value to false, doesn't disable all logging, but only the regular interval logging.
     * If the meter has {@link #setLogOnUpdate(boolean)} enabled, it will still be logged when updated.
     *
     * @param enabled true if the meter should be logged at regular intervals
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder setIntervalLogging(boolean enabled);

    /**
     * Sets the meter to be logged when it's updated.
     * <p>
     * Meter will still be logged at regular intervals if {@link #setIntervalLogging(boolean)} is enabled.
     * </p>
     *
     * @param logOnUpdate true if the meter should be logged when updated
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder setLogOnUpdate(boolean logOnUpdate);

    /**
     * Adds additional condition for logging the meter.Meter will be logged only when this condition is met.
     * <p>
     *     All conditions must be met to log the meter.
     * </p>
     *
     * @param shouldLog predicate returning true if the meter should be logged
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder logWhen(Predicate<IpdMeter> shouldLog);

    /**
     * Sets the listener that will be called when the meter value is updated and {@link #setLogOnUpdate(boolean)} is enabled.
     * <p>
     *     By default this value is set to {@link com.atlassian.diagnostics.ipd.api.IpdConstants#NO_OP}. <br/>
     *     However, products should set this value for all meters to {@link com.atlassian.diagnostics.ipd.api.IpdLoggingService#logMetric(IpdMeter)}.
     * </p>
     * @param updateListener listener that is called when the meter value is updated
     * @return this MeterConfigBuilder
     */
    MeterConfigBuilder setUpdateListener(final Consumer<IpdMeter> updateListener);

}
