package com.atlassian.diagnostics.ipd.api.meters;

import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;

import javax.management.ObjectName;
import java.time.Clock;
import java.util.Collections;
import java.util.Map;

/**
 * Base interface for all IPD meters.
 * Implementations of this interface must be thread-safe and make the meter visible in JMX when its enabled and registered.
 *
 * @since 3.0.0
 */
public interface IpdMeter {

    /**
     * Identifies type of the meter.
     * @return Type ID of the meter
     */
    String getTypeId();

    /**
     * @return MeterKey that identifies meter by name and tag values
     */
    MeterKey getMeterKey();

    /**
     * Meter configuration is constructed when meter is created for the first time and is immutable.
     * @return Read-only meter configuration for this meter
     */
    MeterConfig getConfig();

    /**
     * ObjectName of the MBean. Can be used to query this metric value in JMX.
     * @return ObjectName of this metric
     */
    ObjectName getObjectName();

    /**
     * @return false if the meter is closed {@link #close()} or {@link MeterConfig#getEnabledCheck()} check returns false, true otherwise
     */
    boolean isEnabled();

    /**
     * Meter might be enabled, but temporarily hidden in JMX.
     * This might happen when meter value was not set yet or method {@link #hideUntilUpdate()} was called.
     * @return true if the meter is visible in JMX, false otherwise
     */
    boolean isVisible();

    /**
     * Hides the meter in JMX until the next value update.
     * {@link #isVisible()} will return false until the next value update.
     */
    void hideUntilUpdate();

    /**
     * Returns the timestamp of the last value update in milliseconds.
     * See {@link Clock#millis()}.
     * @return timestamp of the last value update in milliseconds
     */
    long lastUpdateMillis();

    /**
     * Returns the attributes of the meter.
     * @param extraAttributes if true, all attributes visible in JMX will be returned, otherwise only the most important ones
     * @return Map of meter attributes, will return empty map if attributes can't be retrieved
     */
    Map<String, Object> getAttributes(boolean extraAttributes);

    /**
     * Permanently closes this instance of the meter.
     * This instance will always return false for {@link #isEnabled()} and {@link #isVisible()}.<br/>
     * This method is called on meter instances that are removed from the {@link com.atlassian.diagnostics.ipd.api.registry.IpdRegistry}
     */
    void close();

    /**
     * Checks if the meter is closed. A closed meter can't be enabled or visible in JMX.
     * It's a terminal state that is triggered when a meter is removed from the IpdRegistry.
     * You can get a new instance of the meter from the {@link com.atlassian.diagnostics.ipd.api.registry.IpdRegistry}.
     *
     * @return true if the meter is closed, false otherwise
     */
    boolean isClosed();

    /**
     * Noop implementation of {@link IpdMeter}. Does nothing. Meter is won't be visible in JMX.
     */
    abstract class NoopMeter extends AbstractIpdMeter {

        protected NoopMeter(final MeterConfig meterConfig) {
            super(meterConfig);
        }

        @Override
        protected void registerMBean() {
            // Do nothing
        }

        @Override
        protected void unregisterMBean() {
            // Do nothing
        }

        @Override
        public boolean isVisible() {
            return false;
        }

        @Override
        public Map<String, Object> getAttributes(final boolean extraAttributes) {
            return Collections.emptyMap();
        }
    }
}
