package com.atlassian.diagnostics.ipd.api.meters;

import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;

import javax.management.ObjectName;
import java.time.Clock;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;

import static com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig.Properties.CUSTOM_CLOCK;

/**
 * Base class for all IPD meters. Shares common method implementations.
 */
public abstract class AbstractIpdMeter implements IpdMeter {

    private final MeterConfig meterConfig;
    private boolean closed = false;
    private final AtomicLong lastUpdateMillis = new AtomicLong(0);
    private final Clock clock;
    protected AbstractIpdMeter(final MeterConfig meterConfig) {
        this.meterConfig = meterConfig;
        clock = Optional.ofNullable(meterConfig.getProperties().get(CUSTOM_CLOCK))
                .map(Clock.class::cast)
                .orElse(Clock.systemUTC());
    }

    @Override
    public MeterConfig getConfig() {
        return meterConfig;
    }

    @Override
    public MeterKey getMeterKey() {
        return meterConfig.getMeterKey();
    }

    @Override
    public ObjectName getObjectName() {
        return meterConfig.getObjectName();
    }

    @Override
    public boolean isEnabled() {
        return !closed && meterConfig.getEnabledCheck().get();
    }

    @Override
    public void hideUntilUpdate() {
        unregisterMBean();
    }

    @Override
    public long lastUpdateMillis() {
        return lastUpdateMillis.get();
    }

    protected void metricUpdated() {
        registerMBean();
        lastUpdateMillis.set(clock.millis());
        meterConfig.getMetricUpdateListener().accept(this);
    }

    /**
     * Registers the MBean in the MBeanServer.<br/>
     * Method will be called when the meter is enabled and is visible in JMX.
     */
    protected abstract void registerMBean();

    /**
     * Unregisters the MBean from the MBeanServer. <br/>
     * Method will be called when the meter is closed or is hidden from JMX.
     */
    protected abstract void unregisterMBean();

    public void close() {
        unregisterMBean();
        closed = true;
    }

    public boolean isClosed() {
        return closed;
    }
}
