package com.atlassian.diagnostics.ipd.api;

import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfigBuilder;
import com.atlassian.diagnostics.ipd.api.meters.config.ProductMeterConfigBuilder;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;

/**
 * @since 5.0.0
 */
public class MeterConfigurationsImpl implements MeterConfigurations {

    private final PrefixNode rootPrefixTrieNode = new PrefixNode();
    private final Consumer<ProductMeterConfigBuilder> defaultConfig;

    public MeterConfigurationsImpl(Consumer<ProductMeterConfigBuilder> defaultConfig) {
        this.defaultConfig = defaultConfig;
    }

    @Override
    public MeterConfigurations addMeterConfig(final String prefixMatcher, final Consumer<MeterConfigBuilder> meterConfig) {
        rootPrefixTrieNode.setMeterConfig(prefixMatcher, meterConfig);
        return this;
    }

    @Override
    public void evaluateConfig(final MeterKey meterKey, final ProductMeterConfigBuilder configBuilder) {
        defaultConfig.accept(configBuilder);
        rootPrefixTrieNode.evaluateConfig(meterKey.getName(), configBuilder);
    }

    /**
     * Custom trie implementation to store and evaluate meter configurations based on the prefix of the meter key.
     */
    private static class PrefixNode {
        private final AtomicReference<Consumer<MeterConfigBuilder>> meterConfig = new AtomicReference<>(null);
        private final Map<String, PrefixNode> children = new ConcurrentHashMap<>();

        /**
         * Creates a tree of PrefixNodes based on the provided prefix segments, and applies configuration to the last node. Runs recursively.
         * <p>
         *     For prefix "a.b.c" it creates nodes a->b->c and saves the configuration to node c.
         *     Nodes won't be created if they already exist.
         * </p>
         * @param prefix the prefix of the meter key to match
         * @param meterConfig the configuration that will be applied to meters with keys matching the prefix
         */
        public void setMeterConfig(final String prefix, final Consumer<MeterConfigBuilder> meterConfig) {
            if (prefix.isEmpty()) {
                this.meterConfig.set(meterConfig);
            } else {
                children.computeIfAbsent(getFirstSegment(prefix), ignored -> new PrefixNode())
                        .setMeterConfig(removeFirstSegment(prefix), meterConfig);
            }
        }

        /**
         * Traverses the tree of segments for a given prefix and applies them to the provided ProductMeterConfigBuilder.
         * Segments are evaluated from the most generic to the most specific, it is from left to right in the prefix string.
         *
         * @param prefix the prefix of the meter key to match
         * @param configBuilder the builder to apply the configuration to
         */
        public void evaluateConfig(final String prefix, final MeterConfigBuilder configBuilder) {
            final Consumer<MeterConfigBuilder> currentConfig = meterConfig.get();
            if (currentConfig != null) {
                currentConfig.accept(configBuilder);
            }
            if (prefix.isEmpty()) {
                return;
            }
            final PrefixNode child = children.get(getFirstSegment(prefix));
            if (child != null) {
                child.evaluateConfig(removeFirstSegment(prefix), configBuilder);
            }
        }

        /**
         * Returns prefix without first segment.
         * <p>
         *     Example: "a.b.c" -> "b.c"
         * </p>
         * @return Prefix without first segment. Empty string if there are no more segments.
         */
        private String removeFirstSegment(final String prefix) {
            final int index = prefix.indexOf('.');
            if (index == -1) {
                return "";
            }
            return prefix.substring(index + 1);
        }

        /**
         * Returns first segment of the prefix.
         * <p>
         *     Example: "a.b.c" -> "a"
         * </p>
         * @return First segment of the prefix.
         */
        private String getFirstSegment(final String prefix) {
            final int index = prefix.indexOf('.');
            if (index == -1) {
                return prefix;
            }
            return prefix.substring(0, index);
        }
    }
}


