package com.atlassian.diagnostics.internal.ipd.metrics;

import com.atlassian.diagnostics.ipd.api.meters.AbstractIpdMeter;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;
import com.atlassian.util.profiling.MetricKey;
import com.atlassian.util.profiling.Metrics;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.Attribute;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.atlassian.diagnostics.internal.ipd.IpdUtils.getProfilingKey;
import static java.lang.management.ManagementFactory.getPlatformMBeanServer;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toMap;

/**
 * @since 3.0.0
 */
abstract class IpdProfilingMeter extends AbstractIpdMeter {

    private static final Logger LOG = LoggerFactory.getLogger(IpdProfilingMeter.class);
    protected final MBeanServer mBeanServer;
    protected final AtomicBoolean jmxRegistered;
    private final String[] allAttributes;
    private final String[] shortAttributes;
    protected final MetricKey profilingKey;

    protected IpdProfilingMeter(final MeterConfig config,
                                final List<String> allAttributes,
                                final List<String> shortAttributes) {
        this(config, getPlatformMBeanServer(), allAttributes, shortAttributes);
    }

    protected IpdProfilingMeter(final MeterConfig config,
                                final MBeanServer mBeanServer,
                                final List<String> allAttributes,
                                final List<String> shortAttributes) {
        super(config);
        this.mBeanServer = mBeanServer;
        this.allAttributes = allAttributes.toArray(new String[0]);
        this.shortAttributes = shortAttributes.toArray(new String[0]);
        this.jmxRegistered = new AtomicBoolean(false);
        this.profilingKey = getProfilingKey(getMeterKey());
    }

    @Override
    public Map<String, Object> getAttributes(final boolean extraAttributes) {
        return getAttributes(extraAttributes, getObjectName());
    }

    protected Map<String, Object> getAttributes(final boolean extraAttributes, final ObjectName sourceObjectName){
        if (!jmxRegistered.get()) {
            return emptyMap();
        }
        try {
            return mBeanServer.getAttributes(sourceObjectName, extraAttributes ? allAttributes : shortAttributes)
                    .asList()
                    .stream()
                    .filter(Objects::nonNull)
                    .collect(toMap(IpdProfilingMeter::getKey, IpdProfilingMeter::getValue));
        } catch (InstanceNotFoundException e) {
            return emptyMap();
        } catch (ReflectionException e) {
            LOG.error(String.format("Couldn't read ATTRIBUTES for metric %s", getMeterKey()), e);
            return emptyMap();
        }
    }

    private static String getKey(final Attribute attribute) {
        return String.format("_%s", StringUtils.uncapitalize(attribute.getName()));
    }

    private static String getValue(final Attribute attribute) {
        return String.valueOf(attribute.getValue());
    }

    @Override
    public boolean isVisible() {
        return isEnabled() && jmxRegistered.get();
    }

    @Override
    public void unregisterMBean() {
        if (!jmxRegistered.compareAndSet(true, false)) {
            return;
        }
        try {
            Metrics.resetMetric(profilingKey);
            LOG.debug("Unregistering metric: {}", getMeterKey());
        } catch (Exception e) {
            LOG.error("Couldn't unregister metric: {} due to error.", getMeterKey(), e);
        }
    }

    @Override
    public void registerMBean() {
        if (!isEnabled()) {
            return;
        }
        jmxRegistered.set(true);
    }
}
