package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.ipd.api.MeterConfigurations;
import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfigBuilder;
import com.atlassian.diagnostics.ipd.api.meters.config.ProductMeterConfigBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PreDestroy;
import java.util.function.Consumer;

/**
 * Implementation created by {@link DefaultPluginIpdFactory} that wraps an existing {@link IpdMeterRegistry} for a specific plugin.
 * Assures that meters registered by the plugin are removed when the plugin is disabled.
 * @since 5.0.0
 */
public class PluginAwareIpdRegistry extends IpdMeterRegistry {
    private static final String PLUGIN_KEY = "pluginKey";
    private static final Logger LOG = LoggerFactory.getLogger(PluginAwareIpdRegistry.class);
    private final IpdMeterRegistry delegate;
    private final String pluginName;

    public PluginAwareIpdRegistry(final String pluginName,
                                  final IpdMeterRegistry delegate,
                                  final MeterConfigurations pluginMeterConfigurations) {
        super(new DelegateMeterConfigurations(delegate.getMeterConfigurations(), pluginMeterConfigurations), delegate.getMetersStorage());
        this.pluginName = pluginName;
        this.delegate = delegate;
        pluginMeterConfigurations.addMeterConfig("", b -> b.setProperty(PLUGIN_KEY, pluginName));
    }

    @PreDestroy
    public void preDestroy() {
        shutdown();
    }

    @Override
    public void shutdown() {
        delegate.removeIf(meter -> pluginName.equals(meter.getConfig().getProperties().get(PLUGIN_KEY)));
        LOG.info("Unregistering ipd meters for plugin: {}", pluginName);
    }

    private static class DelegateMeterConfigurations implements MeterConfigurations {

        private final MeterConfigurations mainConfiguration;
        private final MeterConfigurations extraConfiguration;

        public DelegateMeterConfigurations(MeterConfigurations mainConfiguration,
                                           MeterConfigurations extraConfiguration) {
            this.mainConfiguration = mainConfiguration;
            this.extraConfiguration = extraConfiguration;
        }

        @Override
        public MeterConfigurations addMeterConfig(final String prefixMatcher, final Consumer<MeterConfigBuilder> meterConfig) {
            extraConfiguration.addMeterConfig(prefixMatcher, meterConfig);
            return this;
        }

        @Override
        public void evaluateConfig(final MeterKey meterKey, final ProductMeterConfigBuilder configBuilder) {
            mainConfiguration.evaluateConfig(meterKey, configBuilder);
            extraConfiguration.evaluateConfig(meterKey, configBuilder);
        }
    }
}
