package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.ipd.api.jobs.IpdJob;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJobRegistration;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJobRegistry;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJobRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.PreDestroy;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Provides default implementation of {@link IpdJobRunner} interface.
 * Product must extend it and implement at least one method {@code isIpdFeatureFlagEnabled();}
 *
 * @since 2.2.0
 */
public class PluginAwareIpdJobRegistry implements IpdJobRegistry {
    private static final Logger log = LoggerFactory.getLogger(PluginAwareIpdJobRegistry.class);

    protected final Set<Class<? extends IpdJob>> jobs = ConcurrentHashMap.newKeySet();
    private final IpdJobRegistry delegate;

    protected PluginAwareIpdJobRegistry(IpdJobRegistry delegate) {
        this.delegate = delegate;
    }

    @Override
    public void register(@Nonnull IpdJob job, @Nonnull String key) {
        register(IpdJobRegistration.builder(job, key).build());
    }

    @Override
    public void register(@Nonnull IpdJobRegistration jobRegistration) {
        jobs.add(jobRegistration.getIpdJob().getClass());
        delegate.register(jobRegistration);
    }

    @Override
    public void unregister(@Nonnull Class<? extends IpdJob> jobClass) {
        jobs.remove(jobClass);
        delegate.unregister(jobClass);
    }

    @PreDestroy
    public void preDestroy() {
        shutdown();
    }

    @Override
    public void shutdown() {
        for (Class<? extends IpdJob> jobClass : jobs) {
            delegate.unregister(jobClass);
        }
        jobs.clear();
    }

}
