package com.atlassian.diagnostics.internal.ipd.metrics;

import com.atlassian.diagnostics.internal.ipd.IpdMetricBuilder;
import com.atlassian.diagnostics.internal.ipd.IpdMetricTypeVerifier;
import com.atlassian.diagnostics.internal.ipd.exceptions.IpdCustomMetricRegisterException;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetricValue;
import com.atlassian.diagnostics.ipd.internal.spi.MetricOptions;
import com.atlassian.util.profiling.MetricTag.RequiredMetricTag;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;

import javax.management.MBeanServer;
import javax.management.MXBean;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

import static java.lang.management.ManagementFactory.getPlatformMBeanServer;
import static java.util.Collections.emptyList;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Metric with custom attributes.
 * @param <T> Type defining metric attributes. This type has to implement interface annotated with @MXBean.
 * @since 3.0.0
 */
public class IpdCustomMetric<T> extends AbstractIpdMetric {

    private static final Logger LOG = getLogger(IpdCustomMetric.class);
    private static final String SINGLE_MX_BEAN_REQUIRED_EXCEPTION = "IpdCustomMetric requires the Type %s to implement exactly one interface with @MXBean annotation.";
    private static final TypeReference<Map<String, Object>> MAP_TYPE_REFERENCE = new TypeReference<Map<String, Object>>(){};

    private final Class<?> dataType;
    private final ObjectMapper mapper;
    private final MBeanServer mBeanServer;
    private final Consumer<IpdMetric> updateListener;
    private final Map<String, String> immutableTags;
    private final AtomicBoolean jmxRegistered = new AtomicBoolean(false);
    private final Class<?> mBeanInterface;
    private final T mBean;

    protected IpdCustomMetric(T mBean, ObjectMapper mapper, MBeanServer mBeanServer, MetricOptions options) {
        super(options);
        this.dataType = mBean.getClass();
        this.mapper = mapper;
        this.mBeanServer = mBeanServer;
        this.mBean = mBean;
        this.updateListener = options.getMetricUpdateListener();
        this.immutableTags = readTags(getObjectName());
        this.mBeanInterface = findMBeanInterface(dataType);
        if (isEnabled()) {
            registerMBean();
        }
    }

    private Class<?> findMBeanInterface(Class<?> type) {
        return Arrays.stream(type.getInterfaces())
                .filter(i -> i.isAnnotationPresent(MXBean.class))
                .reduce((a, b) -> {
                    // If there is more than one MXBean interface
                    throw new IpdCustomMetricRegisterException(String.format(SINGLE_MX_BEAN_REQUIRED_EXCEPTION, type.getName()));
                })
                .orElseThrow(() -> new IpdCustomMetricRegisterException(String.format(SINGLE_MX_BEAN_REQUIRED_EXCEPTION, type.getName())));
    }

    public T getMBeanObject() {
        return mBean;
    }

    protected Map<String, Object> readAttributes() throws IOException {
        // This makes sure we serialize only properties from the MBean interface
        String attributes = mapper.writerWithType(mBeanInterface).writeValueAsString(mBean);
        return mapper.readValue(attributes, MAP_TYPE_REFERENCE);
    }

    @Override
    public List<IpdMetricValue> readValues(boolean extraAttributes) {
        try {
            if (isEnabled()) {
                // Make sure metric is registered, as it's value can be updated directly on mBean object skipping call to IpdCustomMetric::update method.
                registerMBean();
            }
            return ImmutableList.of(new IpdMetricValue(getMetricKey().getMetricName(), getObjectName().getCanonicalName(), immutableTags, readAttributes()));
        } catch (Exception e) {
            LOG.error(String.format("Couldn't read values for Custom IPD metric for metric %s of type %s", getMetricKey(), dataType.getName()), e);
            return emptyList();
        }
    }

    private void registerMBean() {
        if (!jmxRegistered.compareAndSet(false, true)) {
            return;
        }
        try {
            mBeanServer.registerMBean(mBean, getObjectName());
        } catch (Exception e) {
            throw new IpdCustomMetricRegisterException("Exception occurred while registering MBean for IpdCustomMetric with type " + dataType.getName(), e);
        }
    }

    @Override
    public void unregisterJmx() {
        try {
            if (jmxRegistered.compareAndSet(true, false)) {
                mBeanServer.unregisterMBean(getObjectName());
            }
        } catch (Exception e) {
            throw new IpdCustomMetricRegisterException(String.format("Failed to unregister metric %s of type %s", getMetricKey(), dataType));
        }
    }

    public void update(Consumer<T> updater) {
        if (!isEnabled()) {
            return;
        }
        registerMBean();

        updater.accept(mBean);
        updateListener.accept(this);
    }

    /**
     * Creates a custom metric builder.
     * @param metricName metric name
     * @param type type has to implement interface annotated with <b>@MXBean</b>. Getters in this interface will define metric attributes.
     * @param staticTags metric tags
     * @return Metric builder
     * @param <T> Type defining metric attributes
     */
    public static <T> IpdMetricBuilder<IpdCustomMetric<T>> builder(String metricName, Class<T> type, RequiredMetricTag... staticTags) {
        T mBeanObject;
        try {
            mBeanObject = type.getDeclaredConstructor().newInstance();
        } catch (Exception e) {
            throw new IpdCustomMetricRegisterException(String.format("Couldn't create instance with default constructor for metric %s of type %s", metricName, type.getName()), e);
        }

        return builder(metricName, mBeanObject, staticTags);
    }

    /**
     * Creates a custom metric builder with a given object. All value changes in the given object will be reflected in JMX and log file.
     * @param metricName metric name
     * @param object object class has to implement interface annotated with <b>@MXBean</b>. Getters in this interface will define metric attributes.
     * @param staticTags metric tags
     * @return Metric builder
     * @param <T> Type defining metric attributes
     */
    public static <T> IpdMetricBuilder<IpdCustomMetric<T>> builder(String metricName, T object, RequiredMetricTag... staticTags) {
        return new IpdMetricBuilder<>(
                appendToMetricName(metricName,"custom"),
                Arrays.asList(staticTags),
                options -> create(object, options),
                getMetricTypeVerifier(object.getClass()));
    }

    public static <T> IpdCustomMetric<T> create(T object, MetricOptions options) {
        return new IpdCustomMetric<>(object, new ObjectMapper(), getPlatformMBeanServer(), options);
    }

    private static IpdMetricTypeVerifier getMetricTypeVerifier(Class<?> mBeanType) {
        return ipdMetric -> {
            if (!(ipdMetric instanceof IpdCustomMetric)) {
                throw new ClassCastException(String.format("Metric type was %s, but expected %s", ipdMetric.getClass(), IpdCustomMetric.class));
            }
            Class<?> thatMbeanType = ((IpdCustomMetric<?>)ipdMetric).getMBeanObject().getClass();
            if (!thatMbeanType.equals(mBeanType)) {
                throw new ClassCastException(String.format("Metric type was IpdCustomMetric, but the mBean type was different: %s, but expected %s", thatMbeanType, mBeanType));
            }
        };
    }
}
