package com.atlassian.diagnostics.internal.ipd.metrics;

import com.atlassian.diagnostics.internal.ipd.IpdMetricBuilder;
import com.atlassian.diagnostics.internal.ipd.exceptions.IpdRegisterException;
import com.atlassian.diagnostics.internal.jmx.ReadOnlyProxyMBean;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.diagnostics.ipd.internal.spi.MetricOptions;
import com.atlassian.util.profiling.MetricTag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.DynamicMBean;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import java.util.Arrays;
import java.util.List;

import static java.lang.management.ManagementFactory.getPlatformMBeanServer;

/**
 * Metric that copies attributes from another JMX bean.
 *
 * @since 3.3.0
 */
public class IpdCopyMetric extends IpdMicrometerMetric {

    private static final Logger LOG = LoggerFactory.getLogger(IpdCopyMetric.class);

    private final ObjectName objectToCopy;
    private final DynamicMBean jmxBean;

    protected IpdCopyMetric(MetricOptions options,
                            MBeanServer mBeanServer,
                            ObjectName objectToCopy,
                            List<String> allAttributes,
                            List<String> shortAttributes) {
        super(options, mBeanServer, allAttributes, shortAttributes);
        this.objectToCopy = objectToCopy;
        this.jmxBean = new ReadOnlyProxyMBean(objectToCopy, mBeanServer);
        ensureJmxBeanIsRegistered();
    }

    private void ensureJmxBeanIsRegistered() {
        if (mBeanServer.isRegistered(getObjectName())) {
            jmxRegistered.set(true);
        } else {
            synchronized (this) {
                if (!jmxRegistered.get()) {
                    try {
                        mBeanServer.registerMBean(jmxBean, getObjectName());
                        jmxRegistered.set(true);
                    } catch (Exception e) {
                        throw new IpdRegisterException(String.format(
                                "Unable to register JMX bean for metric %s", getMetricKey().getMetricName()), e);
                    }
                }
            }
        }
    }

    private boolean sourceBeanExists() {
        return mBeanServer.isRegistered(objectToCopy);
    }

    public void update() {
        if (!sourceBeanExists()) {
            unregisterJmx();
            removeJmxBeanIfExists();
        } else {
            ensureJmxBeanIsRegistered();
        }
    }

    private void removeJmxBeanIfExists() {
        try {
            if (mBeanServer.isRegistered(getObjectName())) {
                mBeanServer.unregisterMBean(getObjectName());
            }
        } catch (Exception e) {
            LOG.warn("Unable to unregister JMX bean for metric {}", getMetricKey().getMetricName(), e);
        }
    }

    @Override
    protected ObjectName getDataSourceObjectName() {
        return objectToCopy;
    }

    @Override
    public void unregisterJmx() {
        if (!jmxRegistered.compareAndSet(true, false)) {
            return;
        }
        removeJmxBeanIfExists();
    }

    /**
     * Creates a builder for {@link IpdCopyMetric}.
     *
     * @param metricName       Name of the metric.
     * @param objectNameToCopy Object name of the JMX bean to copy attributes from.
     * @param allAttributes    List of attributes to read in verbose mode
     * @param shortAttributes  List of attributes to read in concise mode
     * @param staticTags       Tags that will be added to every metric value.
     * @return Builder for {@link IpdCopyMetric}.
     */
    public static IpdMetricBuilder<IpdCopyMetric> builder(final String metricName,
                                                          final ObjectName objectNameToCopy,
                                                          final List<String> allAttributes,
                                                          final List<String> shortAttributes,
                                                          final MetricTag.RequiredMetricTag... staticTags) {
        return new IpdMetricBuilder<>(
                metricName,
                Arrays.asList(staticTags),
                options -> create(objectNameToCopy, options, allAttributes, shortAttributes),
                IpdCopyMetric::verifyExpectedMetricType);
    }

    private static IpdCopyMetric create(final ObjectName objectToCopy,
                                        final MetricOptions options,
                                        final List<String> allAttributes,
                                        final List<String> shortAttributes) {
        return new IpdCopyMetric(options, getPlatformMBeanServer(), objectToCopy, allAttributes, shortAttributes);
    }

    private static void verifyExpectedMetricType(final IpdMetric ipdMetric) throws ClassCastException {
        if (ipdMetric instanceof IpdCopyMetric) {
            return;
        }
        throw new ClassCastException(String.format("Metric type was %s, but expected %s", ipdMetric.getClass(), IpdCopyMetric.class));
    }
}
