package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.diagnostics.internal.ipd.metrics.IpdCounterMetric;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.diagnostics.ipd.internal.spi.MetricFactory;
import com.atlassian.diagnostics.ipd.internal.spi.MetricOptions;
import com.atlassian.util.profiling.MetricKey;
import com.atlassian.util.profiling.MetricTag.RequiredMetricTag;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

/**
 * <b>Mutable</b> class for building IPD Metrics. Creating this object with a constructor is <b>discouraged</b>.
 * Instead, create builder for specific metric type e.g.
 * <pre>
 *     IpdCounterMetric counter = ipdMainRegistry.register(IpdCounterMetric.builder("name"));
 * </pre>
 * or build metrics directly with registry e.g.
 * <pre>
 *     IpdCounterMetric counter = ipdMainRegistry.counterMetric("name");
 * </pre>
 *
 * <p>
 *      Objects of this class are often mutated with registries {@link IpdMainRegistry#createRegistry(String, RequiredMetricTag...)}
 * </p>
 * @param <T> Type of IpdMetric implementation
 * @since 3.0.0
 */
public class IpdMetricBuilder<T extends IpdMetric> {

    private final Set<RequiredMetricTag> tags = new HashSet<>();
    private MetricFactory<T> metricFactory;
    private String metricName;
    private final IpdMetricTypeVerifier metricTypeVerifier;
    private boolean workInProgressMetric = false;
    private boolean logOnUpdate = false;

    /**
     * Creating this builder object directly with a constructor is <b>discouraged</b>.
     * Instead, use helper methods for each metric type e.g. {@link IpdCounterMetric#builder(String, RequiredMetricTag...)}
     * @param metricName metric name
     * @param tags metric tags
     * @param metricFactory factory that creates metric implementation
     * @param metricTypeVerifier metric type verifier that checks if given {@link IpdMetric} is of expected type by this builder
     */
    public IpdMetricBuilder(final String metricName,
                            final Collection<RequiredMetricTag> tags,
                            final MetricFactory<T> metricFactory,
                            final IpdMetricTypeVerifier metricTypeVerifier) {
        this.metricName = metricName;
        this.metricTypeVerifier = metricTypeVerifier;
        this.tags.addAll(tags);
        this.metricFactory = metricFactory;
    }

    @VisibleForTesting
    IpdMetricBuilder(final String metricName,
                            final Collection<RequiredMetricTag> tags,
                            final MetricFactory<T> metricFactory) {
        this(metricName, tags, metricFactory, (ipdMetric -> {}));
    }

    public T buildMetric(final String productPrefix, final Supplier<Boolean> enabledCheck, final Consumer<IpdMetric> loggingConsumer) {
        return metricFactory.createMetric(new MetricOptions(getMetricKey(),
                productPrefix,
                enabledCheck,
                loggingConsumer,
                !logOnUpdate));
    }

    /**
     * Adds a prefix to the metric name.
     * @param prefix prefix to the metric name, shouldn't start or end with a dot character ".".
     * @return this metric builder object
     */
    public IpdMetricBuilder<T> withPrefix(String prefix) {
        if (StringUtils.isEmpty(metricName)) {
            metricName = prefix;
        } else if (!StringUtils.isEmpty(prefix)) {
            metricName = prefix + "." + metricName;
        }
        return this;
    }

    /**
     * Adds tags to the metric definition.
     * @param tags metric tags
     * @return this metric builder object
     */
    public IpdMetricBuilder<T> withTags(RequiredMetricTag... tags) {
        this.tags.addAll(Arrays.asList(tags));
        return this;
    }

    /**
     * Marks metric as "Work in progress". Metric will be enabled only when the work in progress feature flag is enabled.
     * A disabled metric won't be updated, published to JMX or logged to file.
     * @return this metric builder object
     */
    public IpdMetricBuilder<T> asWorkInProgress() {
        workInProgressMetric = true;
        return this;
    }

    /**
     * Marks metric as "log on update". This option will <b>disable</b> regular logging schedule for this metric.
     * Instead, metric value will be logged each time its value is updated.
     * <p>
     *     Logging will happen synchronously after <b>each</b> value update.
     *     Be careful not to use it in hot-paths or for metrics that are updated frequently.
     * </p>
     * @return this metric builder object
     */
    public IpdMetricBuilder<T> logOnUpdate() {
        logOnUpdate = true;
        return this;
    }

    /**
     * Wraps metric factory method.
     * @param metricFactoryWrapper Method that takes and returns {@link MetricFactory}.
     * @return this metric builder object
     */
    public IpdMetricBuilder<T> wrapMetricFactory(UnaryOperator<MetricFactory<T>> metricFactoryWrapper) {
        metricFactory = metricFactoryWrapper.apply(metricFactory);
        return this;
    }

    public MetricKey getMetricKey() {
        return MetricKey.metricKey(metricName, tags);
    }

    public String getMetricName() {
        return metricName;
    }

    public Set<RequiredMetricTag> getTags() {
        return tags;
    }

    public MetricFactory<T> getMetricFactory() {
        return metricFactory;
    }

    public boolean isWorkInProgressMetric() {
        return workInProgressMetric;
    }

    public boolean isLogOnUpdate() {
        return logOnUpdate;
    }

    public void verifyExpectedMetricType(IpdMetric ipdMetric) throws ClassCastException {
        metricTypeVerifier.verifyIpdMetricType(ipdMetric);
    }
}
