package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.internal.ipd.exceptions.IpdRegisterException;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.util.profiling.MetricKey;

import javax.annotation.Nullable;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * IpdMainRegistry is a root registry that allows for registering new ipd metrics in the product.
 * @since 3.0.0
 */
public class IpdMainRegistry implements IpdMetricRegistry {
    public static final Consumer<IpdMetric> EMPTY_LISTENER = metric -> {};
    private final AtomicBoolean ipdEnabled = new AtomicBoolean(true);
    private final AtomicBoolean ipdEnabledWip = new AtomicBoolean(false);
    private final IpdMainRegistryConfiguration configuration;
    protected final Map<MetricKey, IpdMetric> metrics = new ConcurrentHashMap<>();

    public IpdMainRegistry(final IpdMainRegistryConfiguration configuration) {
        this.configuration = configuration;
    }

    /**
     * Register a metric with given metric builder configuration.
     * Existing metric is returned if metric with the name and tags exists, otherwise new metric is registered.
     *
     * If you don't need any special options, consider using registry helper methods instead:
     * <pre>
     *     IpdCounterMetric metric = ipdMainRegistry.counterMetric("name");
     * </pre>
     *
     * @param metricBuilder metric builder with metric options
     * @return existing or new metric of expected type
     * @param <T> expected metric type
     * @throws IpdRegisterException when there is existing registered metric of the same MetricKey but different type
     */
    @Override
    public <T extends IpdMetric> T register(IpdMetricBuilder<T> metricBuilder) {
        final IpdMetric ipdMetric = metrics.computeIfAbsent(metricBuilder.getMetricKey(), k ->
                metricBuilder.buildMetric(configuration.getProductPrefix(),
                        createMetricEnabledCheck(metricBuilder),
                        getUpdateMetricListener(metricBuilder)));
        try {
            metricBuilder.verifyExpectedMetricType(ipdMetric);
            //noinspection unchecked
            return (T) ipdMetric;
        } catch (ClassCastException e) {
            throw new IpdRegisterException(String.format("Ipd metric type check failed for metric %s", ipdMetric.getMetricKey()), e);
        }
    }

    /**
     * Gets an existing metric from main registry.
     * @param metricKey of the existing metric
     * @return existing metric or null if metric doesn't exist in main registry
     */
    @Nullable
    @Override
    public IpdMetric get(final MetricKey metricKey) {
        return metrics.get(metricKey);
    }

    /**
     * Accepts MetricKey that contains full metric name and tags.
     * <p>
     *      Unregisters the metric and removes it from internal storage. Existing metric instance will be permanently disabled.
     *      Registering metric again with the same MetricKey will return new metric instance.
     * </p>
     * @param metricKey of the existing metric
     */
    @Override
    public void remove(MetricKey metricKey) {
        IpdMetric metric = metrics.remove(metricKey);
        if (metric != null) {
            metric.close();
        }
    }

    /**
     * The same method as <pre>remove(ipdMetricBuilder.getMetricKey())</pre>
     *
     * @param ipdMetricBuilder metric builder of a metric
     */
    @Override
    public void remove(IpdMetricBuilder<?> ipdMetricBuilder) {
        remove(ipdMetricBuilder.getMetricKey());
    }

    /**
     * Unregisters all IPD metrics from JMX and removes them from internal storage. Existing metric instances will be permanently disabled.
     */
    @Override
    public void removeAll() {
        metrics.values().forEach(IpdMetric::close);
        metrics.clear();
    }

    /**
     * Unregisters and removes metrics from main registry for metrics matching the given predicate. Existing metric instances will be permanently disabled.
     * @param predicate the predicate returning true if the metric should be removed
     */
    @Override
    public void removeIf(final Predicate<IpdMetric> predicate) {
        metrics.values().stream()
                .filter(predicate)
                .forEach(IpdMetric::close);
        metrics.values().removeIf(predicate);
    }

    /**
     * Returns all registered metrics and updates feature flag state for all metrics.
     * The metrics enabled state may change after calling this method.
     * @return All registered IpdMetric
     */
    public Set<IpdMetric> getMetrics() {
        updateFeatureFlagState();
        return new HashSet<>(metrics.values());
    }

    /**
     * Updates the feature flag state for all metrics and unregisters the disabled metrics.
     */
    public void unregisterAllDisabledMetrics() {
        updateFeatureFlagState();
        getMetrics().stream()
                .filter(ipdMetric -> !ipdMetric.isEnabled())
                .forEach(IpdMetric::unregisterJmx);
    }

    private void updateFeatureFlagState() {
        ipdEnabled.set(configuration.isIpdEnabled());
        ipdEnabledWip.set(configuration.isIpdWipEnabled());
    }

    protected Supplier<Boolean> createMetricEnabledCheck(IpdMetricBuilder<?> metricBuilder) {
        return () -> ipdEnabled.get() &&
                (!metricBuilder.isWorkInProgressMetric() || ipdEnabledWip.get());
    }

    protected Consumer<IpdMetric> getUpdateMetricListener(IpdMetricBuilder<?> metricBuilder) {
        if (metricBuilder.isLogOnUpdate()) {
            return configuration::metricUpdated;
        }
        return EMPTY_LISTENER;
    }
}

