package com.atlassian.diagnostics.internal.analytics;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.PluginDetails;
import com.atlassian.event.api.AsynchronousPreferred;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * Products that bundle {@code atlassian-diagnostics-core} should also add the following to their analytics-whitelist:
 * <pre>
 *     {
 *         "diagnostics.daily.alerts": [
 *             "count",
 *             "epochDay",
 *             {
 *                 "issueSeverity": ["ERROR", "WARNING", "INFO"]
 *             },
 *             "nodeUuids[*]"
 *         ],
 *         "diagnostics.page.detail": [],
 *         "diagnostics.page.overview": []
 *     }
 * </pre>
 * In addition, they should add the following to their global whitelist:
 * <pre>
 *         "diagnostics.daily.alerts": {
 *             "whitelistedAttributes": [
 *                 "issueId",
 *                 "pluginKey",
 *                 "pluginVersion"
 *             ]
 *         }
 * </pre>
 */
@AsynchronousPreferred
@EventName("diagnostics.daily.alerts")
public class DailyAlertCountAnalyticsEvent {

    private final long count;
    private final long epochDay;
    private final Issue issue;
    private final Set<String> nodeUuids;
    private final PluginDetails plugin;

    public DailyAlertCountAnalyticsEvent(long epochDay, Issue issue, PluginDetails pluginDetails, Set<String> nodeUuids,
                                         long count) {
        this.count = count;
        this.epochDay = epochDay;
        this.issue = requireNonNull(issue, "issue");
        this.plugin = requireNonNull(pluginDetails, "pluginDetails");
        this.nodeUuids = ImmutableSet.copyOf(requireNonNull(nodeUuids, "nodeUuids"));
    }

    public long getCount() {
        return count;
    }

    public long getEpochDay() {
        return epochDay;
    }

    @Nonnull
    public String getIssueId() {
        return issue.getId();
    }

    @Nonnull
    public String getIssueSeverity() {
        return issue.getSeverity().name();
    }

    @Nonnull
    public Set<String> getNodeUuids() {
        return nodeUuids;
    }

    @Nonnull
    public String getPluginKey() {
        return plugin.getKey();
    }

    @Nullable
    public String getPluginVersion() {
        return plugin.getVersion();
    }
}
