package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.Component;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.JsonMapper;
import com.atlassian.diagnostics.Severity;
import com.atlassian.sal.api.message.I18nResolver;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

class SimpleIssue implements Issue {

    private final Component component;
    private final String descriptionI18nKey;
    private final IssueId id;
    private final JsonMapper jsonMapper;
    private final Severity severity;
    private final String summaryI18nKey;

    protected final I18nResolver i18nResolver;

    SimpleIssue(I18nResolver i18nResolver, Component component, IssueId id, String summaryI18nKey,
                String descriptionI18nKey, Severity severity, JsonMapper jsonMapper) {
        this.component = requireNonNull(component, "component");
        this.descriptionI18nKey = requireNonNull(descriptionI18nKey, "descriptionKey");
        this.i18nResolver = requireNonNull(i18nResolver, "i18nResolver");
        this.jsonMapper = requireNonNull(jsonMapper, "jsonMapper");
        this.summaryI18nKey = requireNonNull(summaryI18nKey, "summaryKey");
        this.id = requireNonNull(id, "id");
        this.severity = requireNonNull(severity, "severity");
    }

    @Nonnull
    @Override
    public Component getComponent() {
        return component;
    }

    @Nonnull
    @Override
    public String getDescription() {
        return i18nResolver.getText(descriptionI18nKey);
    }

    @Nonnull
    @Override
    public String getId() {
        return id.toString();
    }

    @SuppressWarnings("unchecked")
    @Nonnull
    @Override
    public <T> JsonMapper<T> getJsonMapper() {
        return jsonMapper;
    }

    @Nonnull
    @Override
    public Severity getSeverity() {
        return severity;
    }

    @Nonnull
    @Override
    public String getSummary() {
        return i18nResolver.getText(summaryI18nKey);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof SimpleIssue)) {
            return false;
        }
        SimpleIssue that = (SimpleIssue) o;
        return Objects.equal(id, that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(id);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", id)
                .add("severity", severity)
                .toString();
    }
}

