package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertListener;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;

import static java.util.Objects.requireNonNull;

public class LockFreeAlertPublisher implements AlertPublisher {

    private static final Logger log = LoggerFactory.getLogger(LockFreeAlertPublisher.class);

    private final Executor executor;
    private final ConcurrentMap<String, AlertListener> listeners;
    private final PluginHelper pluginHelper;

    public LockFreeAlertPublisher(Collection<AlertListener> coreListeners, Executor executor,
                                  PluginHelper pluginHelper) {
        this.executor = executor;
        this.pluginHelper = pluginHelper;

        listeners = new ConcurrentHashMap<>();
        coreListeners.forEach(this::internalSubscribe);
    }

    @Override
    public void publish(@Nonnull Alert alert) {
        requireNonNull(alert, "alert");

        listeners.values().forEach(listener -> {
            try {
                executor.execute(new AlertListenerInvoker(alert, listener));
            } catch (RejectedExecutionException e) {
                log.warn("Failed to schedule invocation of AlertListener {}", listener.getClass().getName(), e);
            }
        });
    }

    @Nonnull
    @Override
    public String subscribe(@Nonnull AlertListener listener) {
        requireNonNull(listener, "listener");
        validateCallerIsHostOrSystemPlugin(listener);

        return internalSubscribe(listener);
    }

    @Override
    public boolean unsubscribe(@Nonnull String subscriptionId) {
        return listeners.remove(subscriptionId) != null;
    }

    private String internalSubscribe(AlertListener listener) {
        String subscriptionId = UUID.randomUUID().toString();
        while (listeners.putIfAbsent(subscriptionId, listener) != null) {
            subscriptionId = UUID.randomUUID().toString();
        }
        return subscriptionId;
    }

    private void validateCallerIsHostOrSystemPlugin(AlertListener listener) {
        Bundle bundle = pluginHelper.getCallingBundle().orElseGet(() -> FrameworkUtil.getBundle(listener.getClass()));
        if (bundle != null && pluginHelper.isUserInstalled(bundle)) {
            throw new IllegalArgumentException("User installed plugins cannot register AlertListeners");
        }
    }

    private static class AlertListenerInvoker implements Runnable {

        private final Alert alert;
        private final AlertListener listener;

        private AlertListenerInvoker(Alert alert, AlertListener listener) {
            this.alert = alert;
            this.listener = listener;
        }

        @Override
        public void run() {
            try {
                listener.onAlert(alert);
            } catch (Exception e) {
                log.warn("AlertListener {} failed", listener.getClass().getName(), e);
            }
        }
    }
}
