package com.atlassian.diagnostics.internal.ipd.metrics;

import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetricValue;
import com.atlassian.diagnostics.ipd.internal.spi.MetricOptions;
import com.atlassian.util.profiling.MetricKey;
import com.atlassian.util.profiling.Metrics;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.Attribute;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.lang.management.ManagementFactory.getPlatformMBeanServer;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toMap;

/**
 * @since 3.0.0
 */
abstract class IpdMicrometerMetric extends AbstractIpdMetric {

    private static final Logger LOG = LoggerFactory.getLogger(IpdMicrometerMetric.class);
    protected final MBeanServer mBeanServer;
    protected final AtomicBoolean jmxRegistered;
    private final String[] allAttributes;
    private final String[] shortAttributes;
    private final Consumer<IpdMetric> onUpdateListener;

    protected IpdMicrometerMetric(MetricOptions options,
                                  List<String> allAttributes,
                                  List<String> shortAttributes) {
        this(options, getPlatformMBeanServer(), allAttributes, shortAttributes);
    }

    protected IpdMicrometerMetric(MetricOptions options,
                                  MBeanServer mBeanServer,
                                  List<String> allAttributes,
                                  List<String> shortAttributes) {
        super(options);
        this.mBeanServer = mBeanServer;
        this.onUpdateListener = options.getMetricUpdateListener();
        this.allAttributes = allAttributes.toArray(new String[0]);
        this.shortAttributes = shortAttributes.toArray(new String[0]);
        this.jmxRegistered = new AtomicBoolean(false);
    }

    protected void metricUpdated() {
        jmxRegistered.set(true);
        onUpdateListener.accept(this);
    }

    protected IpdMetricValue readMetricValue(ObjectName objectName, String[] attributes) {
        try {
            Map<String, Object> attributeValues = readAttributes(objectName, attributes);
            Map<String, String> tagValues = readTags(objectName);
            return new IpdMetricValue(getMetricKey().getMetricName(), objectName.getCanonicalName(), tagValues, attributeValues);
        } catch (Exception e) {
            return null;
        }
    }

    protected Map<String, Object> readAttributes(ObjectName objectName, String[] attributes) throws ReflectionException, InstanceNotFoundException {
        return mBeanServer.getAttributes(objectName, attributes)
                .asList()
                .stream()
                .collect(toMap(this::getKey, this::getValue));
    }

    @Override
    public List<IpdMetricValue> readValues(boolean extraAttributes) {
        if (!jmxRegistered.get()) {
            LOG.debug("Couldn't read value for metric {} because it's missing value in JMX", getMetricKey());
            return emptyList();
        }
        try {
            return mBeanServer.queryNames(getDataSourceObjectName(), null).stream()
                    .map(queriedObjectName -> readMetricValue(queriedObjectName, extraAttributes ? allAttributes : shortAttributes))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
        } catch (Exception e) {
            LOG.error(String.format("Couldn't read values for metric %s", getMetricKey()), e);
            return emptyList();
        }
    }

    /**
     * @return ObjectName of the MBean that will be used as a data source for this metric
     */
    protected ObjectName getDataSourceObjectName() {
        return getObjectName();
    }

    private String getKey(Attribute attribute) {
        return String.format("_%s", StringUtils.uncapitalize(attribute.getName()));
    }

    private String getValue(Attribute attribute) {
        return String.valueOf(attribute.getValue());
    }

    @Override
    public void unregisterJmx() {
        if (!jmxRegistered.compareAndSet(true, false)) {
            return;
        }
        MetricKey metricKey = getMetricKey();
        try {
            Metrics.resetMetric(metricKey);
            LOG.debug("Unregistering metric: {}", metricKey);
        } catch (Exception e) {
            LOG.error("Couldn't unregister metric: {} due to error.", metricKey, e);
        }
    }
}
