package com.atlassian.diagnostics.internal.ipd.metrics;

import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.diagnostics.ipd.internal.spi.MetricOptions;
import com.atlassian.util.profiling.MetricKey;
import com.atlassian.util.profiling.MetricTag;
import com.atlassian.util.profiling.micrometer.util.QualifiedCompatibleHierarchicalNameMapper;
import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.Tags;
import io.micrometer.core.instrument.util.HierarchicalNameMapper;
import org.apache.commons.lang3.StringUtils;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import java.util.Collection;
import java.util.Map;
import java.util.function.Supplier;

import static io.micrometer.core.instrument.Meter.Type.OTHER;
import static io.micrometer.core.instrument.config.NamingConvention.dot;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * Abstract implementation of IpdMetric. Implements common functionalities for all types of IpdMetrics.
 * @since 3.0.0
 */
public abstract class AbstractIpdMetric implements IpdMetric {

    private static final String JMX_TAG_PREFIX = "tag.";

    public static final HierarchicalNameMapper DEFAULT_NAME_MAPPER = new QualifiedCompatibleHierarchicalNameMapper();

    private final MetricKey metricKey;
    private final Supplier<Boolean> enabledCheck;
    private final ObjectName objectName;
    private final MetricOptions metricOptions;
    private boolean closed = false;

    protected AbstractIpdMetric(final MetricOptions metricOptions) {

        this.metricKey = metricOptions.getIpdMetricKey();
        this.enabledCheck = metricOptions.getEnabledCheck();
        this.objectName = constructObjectName(metricOptions.getProductPrefix(), metricKey);
        this.metricOptions = metricOptions;
    }

    public MetricKey getMetricKey() {
        return metricKey;
    }

    public MetricOptions getOptions() {
        return metricOptions;
    }

    public boolean isEnabled() {
        return !closed && Boolean.TRUE.equals(enabledCheck.get());
    }

    public void close() {
        unregisterJmx();
        closed = true;
    }

    public ObjectName getObjectName() {
        return objectName;
    }

    private static ObjectName constructObjectName(String productPrefix, MetricKey metricKey) {
        final Meter.Id dummyMeterId = new Meter.Id(metricKey.getMetricName(), getMicrometerTags(metricKey.getTags()), null, null, OTHER);
        String objectName = productPrefix + ":" + DEFAULT_NAME_MAPPER.toHierarchicalName(dummyMeterId, dot);
        try {
            return new ObjectName(objectName);
        } catch(MalformedObjectNameException e) {
            throw new RuntimeException(e);
        }
    }

    private static Tags getMicrometerTags(Collection<MetricTag.RequiredMetricTag> tags) {
        return Tags.of(tags.stream().map(t -> Tag.of(t.getKey(), t.getValue())).collect(toList()));
    }

    protected static String appendToMetricName(String name, String postfix) {
        if (StringUtils.isEmpty(name) || name.endsWith(".")) {
            return name + postfix;
        }
        return name + "." + postfix;
    }

    protected static Map<String, String> readTags(final ObjectName objectName) {
        return objectName.getKeyPropertyList()
                .entrySet()
                .stream()
                .filter(entry -> entry.getKey().startsWith(JMX_TAG_PREFIX))
                .collect(toMap(
                        entry -> entry.getKey().substring(JMX_TAG_PREFIX.length()),
                        Map.Entry::getValue));
    }
}
