package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.internal.ipd.exceptions.IpdRegisterException;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.util.profiling.MetricKey;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @since 3.0.0
 */
public class IpdMetricRegistryDelegate implements IpdMetricRegistry {

    private final IpdMetricRegistry delegate;
    private final Consumer<IpdMetricBuilder<?>> metricBuilderMutation;
    private final Set<MetricKey> registeredMetrics = ConcurrentHashMap.newKeySet();

    IpdMetricRegistryDelegate(final IpdMetricRegistry delegate,
                              final Consumer<IpdMetricBuilder<?>> metricBuilderMutation) {
        this.delegate = delegate;
        this.metricBuilderMutation = metricBuilderMutation;
    }

    /**
     * Accepts MetricKey that contains full metric name and tags.
     * @param metricKey of the existing metric
     */
    @Override
    public void remove(final MetricKey metricKey) {
        registeredMetrics.remove(metricKey);
        delegate.remove(metricKey);
    }

    /**
     * Accepts IpdMetricBuilder that identifies metric within the context of this registrar.
     * @param ipdMetricBuilder can contain only partial metric name that will be extended by the registry options
     */
    @Override
    public void remove(IpdMetricBuilder<?> ipdMetricBuilder) {
        try {
            metricBuilderMutation.accept(ipdMetricBuilder);
            delegate.remove(ipdMetricBuilder);
        } finally {
            // Builder went through all registers and contains final metricKey
            registeredMetrics.remove(ipdMetricBuilder.getMetricKey());
        }
    }

    @Override
    public void removeIf(final Predicate<IpdMetric> predicate) {
        final List<IpdMetric> metrics = registeredMetrics.stream()
                .map(this::get)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        metrics.stream()
                .filter(predicate)
                .map(IpdMetric::getMetricKey)
                .forEach(this::remove);
    }

    /**
     * New metric is registered if metric with given name and tags doesn't exist, otherwise, existing metric is returned.
     * Metric will have all options from this registry.
     *
     * @param ipdMetricBuilder <b>will be mutated and shouldn't be reused</b>
     * @return existing or new metric of expected type
     * @param <T> expected metric type
     * @throws IpdRegisterException when there is existing registered metric of the same MetricKey but different type
     */
    @Override
    public <T extends IpdMetric> T register(final IpdMetricBuilder<T> ipdMetricBuilder) {
        try {
            metricBuilderMutation.accept(ipdMetricBuilder);
            return delegate.register(ipdMetricBuilder);
        } finally {
            // Builder went through all registers and contains final metricKey
            registeredMetrics.add(ipdMetricBuilder.getMetricKey());
        }
    }

    /**
     * Gets an existing metric from registry.
     * @param metricKey of the existing metric
     * @return existing metric or null if metric doesn't exist in this registry
     */
    @Nullable
    @Override
    public IpdMetric get(final MetricKey metricKey) {
        if (registeredMetrics.contains(metricKey)) {
            return delegate.get(metricKey);
        }

        return null;
    }

    /**
     * Removes and unregisters all metrics registered within this registry.
     */
    @Override
    public void removeAll() {
        registeredMetrics.forEach(delegate::remove);
        registeredMetrics.clear();
    }
}
