package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.ipd.internal.spi.IpdLoggingService;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetricValue;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;

import javax.annotation.ParametersAreNonnullByDefault;
import java.time.Instant;
import java.util.Map;

import static com.atlassian.diagnostics.ipd.internal.spi.IpdConstants.IPD_APP_LOGGER_NAME;
import static com.atlassian.diagnostics.ipd.internal.spi.IpdConstants.IPD_DATA_LOGGER_NAME;
import static com.atlassian.diagnostics.ipd.internal.spi.IpdConstants.LOG_LABEL;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * @since 2.2.0
 */
@ParametersAreNonnullByDefault
public class DefaultLoggingService implements IpdLoggingService {
    private static final Logger dataLogger = getLogger(IPD_DATA_LOGGER_NAME);
    private final Logger regularLogger = getLogger(IPD_APP_LOGGER_NAME);
    private static final ObjectMapper objectMapper = new ObjectMapper()
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    @Override
    public void logMetric(IpdMetric metric, boolean includeExtraLogging) {
        logMetric(metric, getCurrentTimestamp(), includeExtraLogging);
    }

    @Override
    public void logMetric(IpdMetric ipdMetric, String timestamp, boolean includeExtraLogging) {
        if (!ipdMetric.isEnabled()) {
            return;
        }
        ipdMetric.readValues(includeExtraLogging).stream()
                .filter(metricValue -> !metricValue.getAttributes().isEmpty())
                .map(metric -> new IpdLogEntry(timestamp, metric, includeExtraLogging))
                .forEach(value -> {
                    try {
                        logData(objectMapper.writeValueAsString(value));
                    } catch (JsonProcessingException e) {
                        regularLogger.warn("Can't serialize Jmx instrument: {}", value);
                    }
                });
    }

    private void logData(final String data) {
        if (dataLogger.isInfoEnabled()) {
            dataLogger.info(formatData(data));
        }
    }

    public static String getCurrentTimestamp() {
        return String.valueOf(Instant.now().getEpochSecond());
    }

    protected String formatData(String data) {
        return LOG_LABEL + " " + data;
    }

    static class IpdLogEntry {
        private final String timestamp;
        private final String label;
        private final String objectName;
        private final Map<String, String> tags;
        private final Object attributes;
        public IpdLogEntry(final String timestamp, final IpdMetricValue metricValue, boolean includeExtraLogging) {

            this.timestamp = timestamp;
            this.label = metricValue.getLabel().toUpperCase();
            this.objectName = includeExtraLogging ? metricValue.getObjectName() : null;
            this.tags = metricValue.getTags().isEmpty() ? null : metricValue.getTags();
            this.attributes = metricValue.getAttributes();
        }

        public String getTimestamp() {
            return timestamp;
        }

        public String getLabel() {
            return label;
        }

        public String getObjectName() {
            return objectName;
        }

        public Map<String, String> getTags() {
            return tags;
        }

        public Object getAttributes() {
            return attributes;
        }
    }
}
