package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.JsonMapper;
import com.atlassian.diagnostics.detail.ThreadDump;
import com.atlassian.diagnostics.internal.detail.SimpleThreadDump;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.Version;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.io.StringWriter;

import static org.apache.commons.lang3.StringUtils.isBlank;

public class JacksonJsonMapper<T> implements JsonMapper<T> {

    private static final Logger log = LoggerFactory.getLogger(JacksonJsonMapper.class);

    private final Class<T> type;

    protected final ObjectMapper objectMapper;

    public JacksonJsonMapper(Class<T> type) {
        this.type = type;

        objectMapper = new ObjectMapper();
        SimpleModule module = new SimpleModule("Atlassian Diagnostics Built-in", new Version(1, 0, 0, null));
        // register a deserializer for ThreadDump to unmarshal it as a SimpleThreadDump
        module.addDeserializer(ThreadDump.class, new JsonDeserializer<ThreadDump>() {
            @Override
            public ThreadDump deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException {
                return jp.readValueAs(SimpleThreadDump.class);
            }
        });
        objectMapper.registerModule(module);
    }

    @Nonnull
    @Override
    public Class<T> getType() {
        return type;
    }

    @Override
    public T parseJson(String json) {
        if (isBlank(json)) {
            return null;
        }

        try {
            return objectMapper.readValue(json, type);
        } catch (IOException e) {
            log.warn("Failed to parse json as {}: {}", type.getName(), e.getMessage(),
                    log.isDebugEnabled() ? e : null);
        }
        return null;
    }

    @Override
    public String toJson(T value) {
        if (value == null) {
            return null;
        }

        StringWriter writer = new StringWriter();
        try (JsonGenerator generator = objectMapper.getJsonFactory().createJsonGenerator(writer)) {
            generator.writeObject(value);
            generator.flush();
        } catch (IOException e) {
            log.warn("Unexpected exception while rendering an event object to a JSON string", e);
        }
        return writer.toString();
    }
}
