package com.atlassian.diagnostics.internal.jmx;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.Issue;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.Date;
import java.util.concurrent.atomic.AtomicLong;

public class IssueAlerts implements IssueAlertsMXBean {

    private final AtomicLong count;
    private final Issue issue;

    private volatile Instant latestTimestamp;

    public IssueAlerts(Issue issue) {
        this.issue = issue;

        count = new AtomicLong();
    }

    @Nonnull
    @Override
    public String getComponent() {
        return issue.getComponent().getName();
    }

    @Override
    public long getCount() {
        return count.get();
    }

    @Nonnull
    @Override
    public String getDescription() {
        return issue.getDescription();
    }

    @Override
    public Date getLatestAlertTimestamp() {
        Instant ts = latestTimestamp;
        return ts == null ? null : new Date(ts.toEpochMilli());
    }

    @Nonnull
    @Override
    public String getSeverity() {
        return issue.getSeverity().name();
    }

    @Override
    public void reset() {
        count.set(0L);
        latestTimestamp = null;
    }

    void onAlert(Alert alert) {
        count.incrementAndGet();
        Instant timestamp = alert.getTimestamp();
        synchronized (this) {
            if (latestTimestamp == null || latestTimestamp.isBefore(timestamp)) {
                latestTimestamp = timestamp;
            }
        }
    }
}
