package com.atlassian.diagnostics.internal.jmx;

import com.atlassian.diagnostics.Alert;

import java.time.Instant;
import java.util.Date;
import java.util.concurrent.atomic.AtomicLong;

public class Alerts implements AlertsMXBean {

    private final AtomicLong error;
    private final AtomicLong info;
    private final AtomicLong warning;

    private volatile Instant latestTimestamp;

    public Alerts() {
        error = new AtomicLong();
        info = new AtomicLong();
        warning = new AtomicLong();
    }

    @Override
    public long getErrorCount() {
        return error.get();
    }

    @Override
    public long getInfoCount() {
        return info.get();
    }

    @Override
    public Date getLatestAlertTimestamp() {
        Instant ts = latestTimestamp;
        return ts == null ? null : new Date(ts.toEpochMilli());
    }

    @Override
    public long getTotalCount() {
        return error.get() + info.get() + warning.get();
    }

    @Override
    public long getWarningCount() {
        return warning.get();
    }

    @Override
    public void reset() {
        error.set(0L);
        info.set(0L);
        warning.set(0L);
        latestTimestamp = null;
    }

    void onAlert(Alert alert) {
        switch (alert.getIssue().getSeverity()) {
            case INFO:
                info.incrementAndGet();
                break;
            case WARNING:
                warning.incrementAndGet();
                break;
            case ERROR:
                error.incrementAndGet();
                break;
        }

        Instant timestamp = alert.getTimestamp();
        synchronized (this) {
            if (latestTimestamp == null || latestTimestamp.isBefore(timestamp)) {
                latestTimestamp = timestamp;
            }
        }
    }
}
