package com.atlassian.diagnostics.internal.ipd.metrics;

import com.atlassian.diagnostics.internal.ipd.IpdMetricBuilder;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.diagnostics.ipd.internal.spi.MetricOptions;
import com.atlassian.util.profiling.MetricTag;
import com.atlassian.util.profiling.MetricTimer;
import com.atlassian.util.profiling.Metrics;
import com.google.common.collect.ImmutableList;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Metric for a statistic value. Contains a number of statistical attributes: Min, Mean, 95thPercentile, 99thPercentile and more.
 *
 * @since 3.0.0
 */
public class IpdStatsMetric extends IpdMicrometerMetric {

    public static final List<String> allAttributes = ImmutableList.of(
            "50thPercentile",
            "75thPercentile",
            "95thPercentile",
            "98thPercentile",
            "999thPercentile",
            "99thPercentile",
            "Count",
            "DurationUnit",
            "FifteenMinuteRate",
            "FiveMinuteRate",
            "Max",
            "Min",
            "Mean",
            "MeanRate",
            "OneMinuteRate",
            "RateUnit",
            "StdDev",
            "Value");

    public static final List<String> shortAttributes = ImmutableList.of(
            "99thPercentile",
            "Count",
            "Max",
            "Min",
            "Mean");

    private final MetricTimer micrometerMetric;

    protected IpdStatsMetric(MetricOptions options) {
        super(options, allAttributes, shortAttributes);
        micrometerMetric = Metrics.metric(getMetricKey().getMetricName()).tags(getMetricKey().getTags()).timer();
    }

    /**
     * Updates the metric with the given value.
     *
     * @param value value for updated statistics. Assumes the value is in milliseconds.
     */
    public void update(final Long value) {
        update(value, TimeUnit.MILLISECONDS);
    }

    /**
     * Updates the metric with the given value. You may specify the time unit of the value.
     *
     * @param value value for updated statistics
     * @param timeUnit time unit of the value
     */
    public void update(final Long value, final TimeUnit timeUnit) {
        if (isEnabled()) {
            micrometerMetric.update(value, timeUnit);
            metricUpdated();
        }
    }

    public static IpdMetricBuilder<IpdStatsMetric> builder(String metricName, MetricTag.RequiredMetricTag... staticTags) {
        return new IpdMetricBuilder<>(
                appendToMetricName(metricName, "statistics"),
                Arrays.asList(staticTags), 
                IpdStatsMetric::new,
                IpdStatsMetric::verifyExpectedMetricType);
    }

    private static void verifyExpectedMetricType(final IpdMetric ipdMetric) throws ClassCastException {
        if (ipdMetric instanceof IpdStatsMetric) {
            return;
        }
        throw new ClassCastException(String.format("Metric type was %s, but expected %s", ipdMetric.getClass(), IpdStatsMetric.class));
    }
}
