package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.internal.ipd.exceptions.IpdRegisterException;
import com.atlassian.diagnostics.internal.ipd.metrics.IpdCounterMetric;
import com.atlassian.diagnostics.internal.ipd.metrics.IpdCustomMetric;
import com.atlassian.diagnostics.internal.ipd.metrics.IpdStatsMetric;
import com.atlassian.diagnostics.internal.ipd.metrics.IpdValueMetric;
import com.atlassian.diagnostics.internal.ipd.metrics.wrapper.IpdValueAndStatsMetricWrapper;
import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;
import com.atlassian.util.profiling.MetricKey;
import com.atlassian.util.profiling.MetricTag.RequiredMetricTag;

import java.util.function.Consumer;

/**
 * Interface for registering and removing Ipd metrics.
 * @since 3.0.0
 */
public interface IpdMetricRegistry {

    /**
     * Accepts MetricKey that contains full metric name and tags.
     * <p>
     *      Unregisters the metric and removes it from internal storage. Existing metric instance will be permanently disabled.
     *      Registering metric again with the same MetricKey will return new metric instance.
     * </p>
     * @param metricKey of the existing metric
     */
    void remove(final MetricKey metricKey);

    /**
     * Accepts IpdMetricBuilder that identifies metric within the context of this registrar.
     * @param ipdMetricBuilder can contain only partial metric name that will be extended by the registry options
     */
    void remove(final IpdMetricBuilder<?> ipdMetricBuilder);

    /**
     * New metric is registered if metric with given name and tags doesn't exist, otherwise, existing metric is returned.
     *
     * @param metricBuilder metric builder with metric options, metricBuilder <b>might be mutated and shouldn't be reused</b>
     * @return existing or new metric of expected type
     * @param <T> expected metric type
     * @throws IpdRegisterException when there is existing registered metric of the same MetricKey but different type
     */
    <T extends IpdMetric> T register(final IpdMetricBuilder<T> metricBuilder);

    /**
     * Unregisters all IPD metrics within this registry from JMX and removes them from internal storage. Existing metric instances will be permanently disabled.
     */
    void removeAll();

    /**
     * Creates a new registry with given metric prefix and tags.
     * Metrics registered through returned registry will have common prefix and tags.
     *
     * @param prefix common name applied to all metrics registered through returned registry, shouldn't start or end with a dot.
     * @param tags common tags applied to all metrics registered through returned registry
     * @return IpdMetricRegistry with common prefix and tags
     */
    default IpdMetricRegistry createRegistry(final String prefix, RequiredMetricTag... tags) {
        return createRegistry(builder -> builder.withPrefix(prefix).withTags(tags));
    }

    /**
     * Creates a new registry with given metric mutation.
     * All metrics registered through returned registry will have applied this IpdMetricBuilder mutation.
     *
     * @param metricBuilderMutation method that accepts IpdMetricBuilder and modifies its properties. This method will apply to each registered metric.
     * @return IpdMetricRegistry that will apply mutation to all registered metrics
     */
    default IpdMetricRegistry createRegistry(final Consumer<IpdMetricBuilder<?>> metricBuilderMutation) {
        return new IpdMetricRegistryDelegate(this, metricBuilderMutation);
    }

    /**
     * Creates metric wrapper containing both value and statistic metrics.
     * @param name metric name
     * @param staticTags metric tags
     * @return Wrapper containing existing or new metrics
     */
    default IpdValueAndStatsMetricWrapper valueAndStatsMetric(final String name, final RequiredMetricTag... staticTags) {
        return new IpdValueAndStatsMetricWrapper(
                register(IpdStatsMetric.builder(name, staticTags)),
                register(IpdValueMetric.builder(name, staticTags)));
    }

    /**
     * Creates value metric.
     * @param name metric name
     * @param staticTags metric tags
     * @return Existing or new metric
     */
    default IpdValueMetric valueMetric(final String name, final RequiredMetricTag... staticTags) {
        return register(IpdValueMetric.builder(name, staticTags));
    }

    /**
     * Creates statistic metric.
     * @param name metric name
     * @param staticTags metric tags
     * @return Existing or new metric
     */
    default IpdStatsMetric statsMetric(final String name, final RequiredMetricTag... staticTags) {
        return register(IpdStatsMetric.builder(name, staticTags));
    }

    /**
     * Creates counter metric.
     * @param name metric name
     * @param staticTags metric tags
     * @return Existing or new metric
     */
    default IpdCounterMetric counterMetric(final String name, final RequiredMetricTag... staticTags) {
        return register(IpdCounterMetric.builder(name, staticTags));
    }

    /**
     * Creates a custom metric with a given type of attributes.
     * @param name metric name
     * @param type type has to implement interface annotated with <b>@MXBean</b>. Getters in this interface will define metric attributes.
     * @param staticTags metric tags
     * @return existing or new metric
     * @param <T> Type defining metric attributes
     */
    default <T> IpdCustomMetric<T> customMetric(final String name, final Class<T> type, final RequiredMetricTag... staticTags) {
        return register(IpdCustomMetric.builder(name, type, staticTags));
    }

    /**
     * Creates a custom metric with given object. All value changes in the given object will be reflected in JMX and log file.
     * @param name metric name
     * @param obj object has to implement interface annotated with <b>@MXBean</b>. Getters in this interface will define metric attributes.
     * @param staticTags metric tags
     * @return existing or new metric
     * @param <T> Type defining metric attributes
     */
    default <T> IpdCustomMetric<T> customMetric(final String name, final T obj, final RequiredMetricTag... staticTags) {
        return register(IpdCustomMetric.builder(name, obj, staticTags));
    }
}
