package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.ipd.internal.spi.IpdMetric;

import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * Default implementation of IpdMainRegistryConfiguration allowing for delayed initialization of properties through setters.
 * @since 3.0.0
 */
public class DefaultIpdMainRegistryConfiguration implements IpdMainRegistryConfiguration {
    private final AtomicReference<Supplier<Boolean>> ipdFfSupplier = new AtomicReference<>(() -> true);

    private final AtomicReference<Supplier<Boolean>> ipdWipFfSupplier = new AtomicReference<>(() -> false);
    private final AtomicReference<Consumer<IpdMetric>> logMetricOnDemand = new AtomicReference<>(metric -> {});
    private final String productPrefix;

    public DefaultIpdMainRegistryConfiguration(final String productPrefix) {
        this.productPrefix = productPrefix;
    }

    public void setFFSupplier(final Supplier<Boolean> fFSupplier) {
        ipdFfSupplier.set(fFSupplier);
    }

    public void setWipFFSupplier(final Supplier<Boolean> wipFFSupplier) {
        ipdWipFfSupplier.set(wipFFSupplier);
    }

    public void setMetricLogOnDemandListener(final Consumer<IpdMetric> logOnDemandListener) {
        this.logMetricOnDemand.set(logOnDemandListener);
    }

    @Override
    public String getProductPrefix() {
        return productPrefix;
    }


    @Override
    public boolean isIpdEnabled() {
        return ipdFfSupplier.get().get();
    }

    @Override
    public boolean isIpdWipEnabled() {
        return ipdWipFfSupplier.get().get();
    }

    @Override
    public void metricUpdated(IpdMetric metric) {
        logMetricOnDemand.get().accept(metric);
    }
}
