package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.ipd.internal.spi.IpdJob;
import com.atlassian.diagnostics.ipd.internal.spi.IpdJobRunner;
import com.atlassian.diagnostics.ipd.internal.spi.IpdFeatureFlagAware;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Provides default implementation of {@link IpdJobRunner} interface.
 * Product must extend it and implement at least one method {@code isIpdFeatureFlagEnabled();}
 *
 * @since 2.2.0
 */
public abstract class DefaultIpdJobRunner
        implements IpdJobRunner, IpdFeatureFlagAware {

    private static final Logger log = LoggerFactory.getLogger(DefaultIpdJobRunner.class);

    protected final Map<Class<?>, IpdJob> jobs;

    protected DefaultIpdJobRunner() {
        this.jobs = new ConcurrentHashMap<>();
    }

    @Override
    public void register(@Nonnull IpdJob job) {
        jobs.put(job.getClass(), job);
        log.info("Job {} has been registered", job.getClass().getName());
    }

    @Override
    public void runJobs() {
        if (!isIpdFeatureFlagEnabled()) {
            log.debug("Not executing IpdJobs because in.product.diagnostics feature flag is disabled");
            return;
        }
        for (IpdJob job : jobs.values()) {
            if (job.isWorkInProgressJob() && !isWipIpdFeatureFlagEnabled()) {
                log.debug("Ignoring job {}, in.product.diagnostics.wip feature flag is not enabled", job.getClass().getName());
                continue;
            }
            log.debug("Running job {}", job.getClass().getName());
            try {
                job.runJob();
            } catch (RuntimeException ex) {
                log.error(String.format("Error during executing %s job", job.getClass().getName()), ex);
            }
        }
    }

}
