package com.atlassian.diagnostics.internal.concurrent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.time.Clock;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;

public class Gate {

    private static final Logger log = LoggerFactory.getLogger(Gate.class);

    private final Clock clock;
    private final Duration minDurationBetweenCalls;
    private final ReentrantLock lock;

    private volatile long nextInvocationTimestamp;

    public Gate(@Nonnull Clock clock, @Nonnull Duration minDurationBetweenCalls) {
        this.clock = requireNonNull(clock, "clock");
        this.minDurationBetweenCalls = requireNonNull(minDurationBetweenCalls);

        lock = new ReentrantLock();
        nextInvocationTimestamp = 0L;
    }

    public Gate(@Nonnull Duration minDurationBetweenCalls) {
        this(Clock.systemDefaultZone(), minDurationBetweenCalls);
    }

    @Nonnull
    public <T> Optional<T> ifAccessible(@Nonnull Supplier<T> operation) {
        long now = clock.millis();
        if (now >= nextInvocationTimestamp) {
            if (lock.tryLock()) {
                try {
                    if (now >= nextInvocationTimestamp) {
                        nextInvocationTimestamp = clock.millis() + minDurationBetweenCalls.toMillis();
                        log.trace("running operation");
                        return ofNullable(operation.get());
                    }
                } finally {
                    lock.unlock();
                }
            }
        }
        log.trace("skipping operation because it already ran in the past {} ms", minDurationBetweenCalls.toMillis());
        return empty();
    }

    @Nonnull
    public void ifAccessible(@Nonnull Runnable operation) {
        long now = clock.millis();
        if (now >= nextInvocationTimestamp) {
            if (lock.tryLock()) {
                try {
                    if (now >= nextInvocationTimestamp) {
                        nextInvocationTimestamp = clock.millis() + minDurationBetweenCalls.toMillis();
                        log.trace("running operation");
                        operation.run();
                    }
                } finally {
                    lock.unlock();
                }
            }
        }
        log.trace("skipping operation because it already ran in the past {} ms", minDurationBetweenCalls.toMillis());
    }
}
