package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertListener;
import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.Severity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.defaultIfBlank;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * AlertListener implementation that writes alerts to 2 logs:
 * <ul>
 *     <li>
 *         A {@link #setDataLogger(Logger) data logger} that writes all alert details in ; separated format, which
 *         is easily machine parsable.
 *     </li>
 *     <li>
 *         A {@link #setRegularLogger(Logger) regular logger} that writes alert details in a human readable format
 *     </li>
 * </ul>
 *
 * The loggers can be configured using the setters. The default configuration uses "atlassian-diagnostics" as the logger
 * name for the regular logger and "atlassian-diagnostics-data-logger" for the data logger.
 */
public class LoggingAlertListener implements AlertListener {

    private static final String LOG_MESSAGE = "{} Component '{}' alerted '{}' (details: {}, trigger: {})";

    // the logger used for persisting alerts in a machine parsable format, to be included in the support zip
    private Logger dataLogger;
    private String dataSeparator;
    // the logger used for nicely formatted messages for the regular log
    private Logger regularLogger;

    public LoggingAlertListener() {
        dataLogger = LoggerFactory.getLogger("atlassian-diagnostics-data-logger");
        dataSeparator = " ; ";
        regularLogger = LoggerFactory.getLogger("atlassian-diagnostics");
    }

    @Override
    public void onAlert(@Nonnull Alert alert) {
        Severity severity = requireNonNull(alert, "alert").getIssue().getSeverity();

        boolean dataLoggerEnabled = isEnabled(dataLogger, severity);
        boolean regularLoggerEnabled = isEnabled(regularLogger, severity);
        if (!dataLoggerEnabled && !regularLoggerEnabled) {
            // no loggers enabled for the alert's severity
            return;
        }

        // generate the json alert details once, so it can be reused for the dataLogger and regularLogger
        String detailsJson = getDetailsJson(alert);
        if (dataLoggerEnabled) {
            log(dataLogger, severity, createDataLoggerMessage(alert, dataSeparator, detailsJson));
        }
        if (regularLoggerEnabled) {
            Issue issue = alert.getIssue();
            log(regularLogger, severity, LOG_MESSAGE, alert.getTimestamp(), issue.getComponent().getName(),
                    issue.getSummary(), defaultIfBlank(detailsJson, "{}"), getTrigger(alert));
        }
    }

    public void setDataLogger(Logger dataLogger) {
        this.dataLogger = dataLogger;
    }

    public void setRegularLogger(Logger regularLogger) {
        this.regularLogger = regularLogger;
    }

    private static void appendProperty(StringBuilder builder, String propertyName, String value) {
        if (isBlank(value)) {
            return;
        }

        if (builder.length() > 0) {
            builder.append(", ");
        }
        builder.append("\"").append(propertyName).append("\": \"").append(value).append("\"");
    }

    private static String createDataLoggerMessage(Alert alert, String separator, String detailsJson) {
        // Format: timestamp | severity | component-id | issue-id | summary | plugin-key | plugin-version | module | details
        Issue issue = alert.getIssue();
        StringBuilder builder = new StringBuilder()
                .append(alert.getTimestamp().toEpochMilli()).append(separator)
                .append(issue.getSeverity()).append(separator)
                .append(issue.getComponent().getId()).append(separator)
                .append(issue.getId()).append(separator)
                .append(issue.getSummary()).append(separator);
        AlertTrigger trigger = alert.getTrigger();
        builder.append(trigger.getPluginKey()).append(separator);
        trigger.getPluginVersion().ifPresent(builder::append);
        builder.append(separator);
        trigger.getModule().ifPresent(builder::append);
        builder.append(separator);
        builder.append(detailsJson);
        return builder.toString();
    }

    private static String getDetailsJson(Alert alert) {
        try {
            return alert.getDetails()
                    .map(details -> alert.getIssue().getJsonMapper().toJson(details))
                    .orElse("");
        } catch (RuntimeException e) {
            return "";
        }
    }

    private static String getTrigger(Alert alert) {
        AlertTrigger trigger = alert.getTrigger();
        StringBuilder builder = new StringBuilder();
        appendProperty(builder, "pluginKey", trigger.getPluginKey());
        trigger.getPluginVersion().ifPresent(key -> appendProperty(builder, "pluginVersion", key));
        trigger.getModule().ifPresent(key -> appendProperty(builder, "module", key));
        return "{" + builder.toString() + "}";
    }

    private static boolean isEnabled(Logger logger, Severity severity) {
        if (logger == null) {
            return false;
        }
        switch (severity) {
            case INFO:
                return logger.isInfoEnabled();
            case WARNING:
                return logger.isWarnEnabled();
            case ERROR:
                return logger.isErrorEnabled();
            default:
                return false;
        }
    }

    private static void log(Logger logger, Severity severity, String format, Object... arguments) {
        switch (severity) {
            case INFO:
                logger.info(format, arguments);
                break;
            case WARNING:
                logger.warn(format, arguments);
                break;
            case ERROR:
                logger.error(format, arguments);
                break;
        }
    }
}
