package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.AlertWithElisions;
import com.atlassian.diagnostics.Interval;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.PageCallback;
import com.atlassian.diagnostics.PageRequest;
import com.atlassian.diagnostics.internal.dao.AlertEntity;
import com.atlassian.diagnostics.internal.dao.MinimalAlertEntity;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Helper class that collects and groups {@link MinimalAlertEntity entities} by (issue, plugin-key, node) tuple for
 * time windows of {@code windowSize}.
 * <p>
 * These groups are then transformed into {@link AlertWithElisions alerts with elision information}, where the most
 * representative alert for the group is used. Each of these {@link AlertWithElisions alerts} is then emitted to the
 * delegate {@link PageCallback callback}.
 * <p>
 */
class AlertWithElisionsCollector {

    private final int end;
    private final List<AlertCandidate> ids;
    private final IssueSupplier issueSupplier;
    private final int start;
    private final long windowMillis;

    private Map<AlertGroupKey, AlertCandidate> curEpochWindowCandidates;
    private long curEpochWindow;
    private int nextRow;

    AlertWithElisionsCollector(IssueSupplier issueSupplier, PageRequest pageRequest, Duration windowSize) {
        this.issueSupplier = issueSupplier;

        curEpochWindowCandidates = new HashMap<>();
        ids = new ArrayList<>(Math.min(256, pageRequest.getLimit() + 1));
        start = pageRequest.getStart();
        end = start + pageRequest.getLimit() + 1;
        windowMillis = windowSize.toMillis();
    }

    void add(@Nonnull MinimalAlertEntity alert) {
        long timestamp = alert.getTimestamp().toEpochMilli();
        long epochWindow = timestamp / windowMillis;
        if (epochWindow != curEpochWindow) {
            nextWindow();
            curEpochWindow = epochWindow;
        }

        AlertGroupKey key = new AlertGroupKey(alert.getIssueId(), alert.getTriggerPluginKey(), alert.getNodeName());
        curEpochWindowCandidates.computeIfAbsent(key, k -> new AlertCandidate())
                .add(alert);
    }

    /**
     * @return the IDs of the samples that best represent the alerts that occurred in the requested page
     */
    @Nonnull
    List<Long> getAlertIdsToLoad() {
        return ids.stream().map(candidate -> candidate.bestId).collect(Collectors.toList());
    }

    /**
     * @return {@code true} if the collector has determined the most representative samples for the requested page.
     *         These samples may not have been loaded yet
     */
    boolean hasCompletePage() {
        return nextRow > end;
    }

    /**
     * Signals to the collector that all alerts that {@link #getAlertIdsToLoad() should be loaded} have been
     * {@link #resolveCandidate(AlertEntity) provided} to the collector. It may be the case that alerts that were
     * previously determined to be the best candidate are no longer available (they have been deleted). If this is the
     * case, these alerts should just be skipped and the remaining alerts should be returned.
     */
    List<AlertWithElisions> onEndAlertResolution() {
        ImmutableList.Builder<AlertWithElisions> builder = ImmutableList.builder();
        Iterator<AlertCandidate> it = ids.iterator();
        while (it.hasNext()) {
            AlertCandidate candidate = it.next();
            if (candidate.resolved != null) {
                builder.add(candidate.resolved);
            }
            it.remove();
        }
        return builder.build();
    }

    /**
     * Signals tot the collector that all {@link MinimalAlertEntity minimal alerts} have been
     * {@link #add(MinimalAlertEntity) provided} to the collector.
     */
    void onEndAlertScan() {
        nextWindow();
    }

    /**
     * Called when one of the {@link #getAlertIdsToLoad() requested alerts} has been loaded. The collector looks up
     * the corresponding {@link AlertCandidate} and resolves it. The collector then returns a list of all now-resolved
     * {@link AlertWithElisions}, maintaining the original order in the {@link #ids} list.
     *
     * @param alert the alert that has been loaded
     * @return a list of {@link AlertWithElisions} that can be provided to the callback
     */
    @Nonnull
    List<AlertWithElisions> resolveCandidate(AlertEntity alert) {
        ids.stream()
                .filter(candidate -> candidate.bestId == alert.getId())
                .findFirst()
                .ifPresent(candidate -> candidate.resolve(alert, windowMillis));

        ImmutableList.Builder<AlertWithElisions> builder = ImmutableList.builder();
        Iterator<AlertCandidate> it = ids.iterator();
        while (it.hasNext()) {
            AlertCandidate candidate = it.next();
            if (candidate.resolved != null) {
                builder.add(candidate.resolved);
                it.remove();
            } else {
                return builder.build();
            }
        }
        return builder.build();
    }

    private void nextWindow() {
        int skip = Math.max(0, start - nextRow);
        int limit = Math.max(0, end - nextRow);
        int size = curEpochWindowCandidates.size();
        if (skip < size && limit > 0) {
            // at least some items are in range, add them to collection
            curEpochWindowCandidates.values().stream().sorted()
                    .skip(skip)
                    .limit(limit)
                    .forEach(ids::add);
        }
        nextRow += size;
        curEpochWindowCandidates.clear();
    }

    private class AlertCandidate implements Comparable<AlertCandidate> {

        private long bestId;
        private int bestLength;
        private int count;
        private Instant latestTimestamp;

        private AlertWithElisions resolved;

        @Override
        public int compareTo(AlertCandidate o) {
            // new to old
            return o.latestTimestamp.compareTo(this.latestTimestamp);
        }

        void add(MinimalAlertEntity minimalAlert) {
            Instant timestamp = minimalAlert.getTimestamp();
            if (latestTimestamp == null || timestamp.isAfter(latestTimestamp)) {
                latestTimestamp = timestamp;
            }
            int detailLength = minimalAlert.getDetailsJsonLength();
            if (bestId == 0 || detailLength > bestLength) {
                bestId = minimalAlert.getId();
                bestLength = detailLength;
            }
            ++count;
        }

        void resolve(AlertEntity alert, long windowMillis) {
            Issue issue = issueSupplier.getIssue(alert.getIssueId(), alert.getIssueSeverity());
            SimpleAlertWithElisions.Builder builder = new SimpleAlertWithElisions.Builder(issue, alert.getNodeName())
                    .id(alert.getId())
                    .detailsAsJson(alert.getDetailsJson())
                    .timestamp(alert.getTimestamp())
                    .trigger(new AlertTrigger.Builder()
                            .plugin(alert.getTriggerPluginKey(), alert.getTriggerPluginVersion())
                            .module(alert.getTriggerModule())
                            .build());
            if (count > 1) {
                long windowStart = alert.getTimestamp().toEpochMilli();
                windowStart -= windowStart % windowMillis;
                Instant start = Instant.ofEpochMilli(windowStart);
                Interval interval = new SimpleInterval(start, start.plusMillis(windowMillis));
                builder.elisions(new SimpleElisions(interval, count - 1));
            }
            resolved = builder.build();
        }
    }

    private static class AlertGroupKey {
        private final String issueId;
        private final String node;
        private final String pluginKey;

        private AlertGroupKey(String issueId, String pluginKey, String node) {
            this.issueId = issueId;
            this.node = node;
            this.pluginKey = pluginKey;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            AlertGroupKey that = (AlertGroupKey) o;
            return Objects.equals(issueId, that.issueId) &&
                    Objects.equals(node, that.node) &&
                    Objects.equals(pluginKey, that.pluginKey);
        }

        @Override
        public int hashCode() {
            return Objects.hash(issueId, node, pluginKey);
        }
    }
}
