package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.AlertCount;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.PluginDetails;
import com.atlassian.diagnostics.internal.dao.AlertMetric;

import java.util.Objects;

class AlertCountCollector {

    private final IssueSupplier issueSupplier;
    private final PluginDetailsSupplier pluginSupplier;

    private volatile SimpleAlertCount.Builder builder;
    private volatile AlertMetric prevRow;

    AlertCountCollector(IssueSupplier issueSupplier, PluginDetailsSupplier pluginSupplier) {
        this.issueSupplier = issueSupplier;
        this.pluginSupplier = pluginSupplier;
    }

    AlertCount onRow(AlertMetric row) {
        AlertCount result = null;
        if (builder != null && !isSameIssueAndPlugin(prevRow, row)) {
            result = maybeEmit();
        }

        if (builder == null) {
            Issue issue = issueSupplier.getIssue(row.getIssueId(), row.getIssueSeverity());
            PluginDetails plugin = pluginSupplier.getPluginDetails(row.getPluginKey(), row.getPluginVersion());
            builder = new SimpleAlertCount.Builder(issue, plugin);
        }
        builder.setCountForNode(row.getNodeName(), row.getCount());
        prevRow = row;

        return result;
    }

    AlertCount onEnd() {
        return maybeEmit();
    }

    private AlertCount maybeEmit() {
        if (builder != null) {
            AlertCount result = builder.build();
            builder = null;
            return result;
        }
        return null;
    }

    private boolean isSameIssueAndPlugin(AlertMetric metric1, AlertMetric metric2) {
        return Objects.equals(metric1.getIssueId(), metric2.getIssueId()) &&
                Objects.equals(metric1.getPluginKey(), metric2.getPluginKey()) &&
                Objects.equals(metric1.getPluginVersion(), metric2.getPluginVersion());
    }
}
