package com.atlassian.diagnostics.internal.jmx;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.management.InstanceNotFoundException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.Query;
import javax.management.ReflectionException;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.ThreadMXBean;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class DefaultJmxService implements JmxService {

    private static final Logger logger = LoggerFactory.getLogger(DefaultJmxService.class);

    @Override
    public ThreadMXBean getThreadMXBean() {
        return ManagementFactory.getThreadMXBean();
    }

    @Override
    @Nonnull
    public List<MemoryPoolMXBean> getMemoryPoolMXBeans() {
        return ManagementFactory.getMemoryPoolMXBeans();
    }

    @Override
    @Nonnull
    public List<GarbageCollectorMXBean> getGarbageCollectorMXBeans() {
        return ManagementFactory.getGarbageCollectorMXBeans();
    }

    @Override
    public boolean hasObjectName(@Nonnull final String instanceOfQuery) {
        return getObjectName(instanceOfQuery) != null;
    }

    @Override
    @Nullable
    public <T> T getJmxAttribute(@Nonnull final String instanceOfQuery, @Nonnull final String attributeName) {
        final List<T> attributes = getJmxAttributes(instanceOfQuery, new String[] { attributeName });
        return !attributes.isEmpty() ? attributes.get(0) : null;
    }

    @Override
    @Nonnull
    public <T> List<T> getJmxAttributes(@Nonnull final String instanceOfQuery, @Nonnull final String[] attributeNames) {
        final ObjectName objectName = getObjectName(instanceOfQuery);
        if (objectName != null) {
            try {
                return ManagementFactory.getPlatformMBeanServer().getAttributes(objectName, attributeNames).asList().stream()
                        .map(attribute -> (T) attribute.getValue())
                        .collect(Collectors.toList());
            } catch (InstanceNotFoundException | ReflectionException | ClassCastException e) {
                logger.debug("Failed to get jmxAttributes", e);
            }
        }

        return Collections.emptyList();
    }

    private ObjectName getObjectName(final String instanceOfQuery) {
        final Set<ObjectInstance> jmxObjectInstances = ManagementFactory.getPlatformMBeanServer().queryMBeans(null, Query.isInstanceOf(Query.value(instanceOfQuery)));
        return jmxObjectInstances.isEmpty()
                ? null
                : jmxObjectInstances.iterator().next().getObjectName();
    }
}
