package com.atlassian.diagnostics.internal.detail;

import com.atlassian.diagnostics.detail.ThreadDump;
import com.google.common.base.MoreObjects;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;
import org.codehaus.jackson.map.annotate.JsonSerialize;

import javax.annotation.Nonnull;
import java.util.Objects;
import java.util.Optional;

import static java.util.Optional.ofNullable;

@JsonSerialize(include = JsonSerialize.Inclusion.NON_EMPTY)
public class SimpleThreadDump implements ThreadDump {

    private final long id;
    private final String name;
    private final boolean daemon;
    private final Thread.State state;
    private final String stackTrace;

    public SimpleThreadDump(Thread thread, String stackTrace) {
        this.stackTrace = stackTrace;

        id = thread.getId();
        name = thread.getName();
        daemon = thread.isDaemon();
        state = thread.getState();
    }

    public SimpleThreadDump(@JsonProperty("id") long id, @JsonProperty("name") String name,
                            @JsonProperty("daemon") boolean daemon, @JsonProperty("state") String state,
                            @JsonProperty("stackTrace") String stackTrace) {
        this.id = id;
        this.name = name;
        this.daemon = daemon;
        this.state = Thread.State.valueOf(state);
        this.stackTrace = stackTrace;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SimpleThreadDump that = (SimpleThreadDump) o;
        return id == that.id &&
                daemon == that.daemon &&
                Objects.equals(name, that.name) &&
                state == that.state &&
                Objects.equals(stackTrace, that.stackTrace);
    }

    @Override
    public long getId() {
        return id;
    }

    @Nonnull
    @Override
    public String getName() {
        return name;
    }

    @Nonnull
    @Override
    public Thread.State getState() {
        return state;
    }

    @JsonIgnore
    @Nonnull
    @Override
    public Optional<String> getStackTrace() {
        return ofNullable(stackTrace);
    }

    @JsonProperty("stackTrace")
    private String getStackTraceRaw() {
        return stackTrace;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, name, daemon, state, stackTrace);
    }

    @Override
    public boolean isDaemon() {
        return daemon;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", id)
                .add("name", name)
                .add("daemon", daemon)
                .add("state", state)
                .add("stackTrace", stackTrace)
                .toString();
    }
}
