package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.PluginDetails;
import com.atlassian.diagnostics.util.CallingBundleResolver;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.metadata.PluginMetadataManager;
import com.atlassian.plugin.osgi.util.OsgiHeaderUtil;
import com.atlassian.sal.api.message.I18nResolver;
import org.osgi.framework.Bundle;

import javax.annotation.Nonnull;
import java.util.Optional;

import static com.atlassian.diagnostics.DiagnosticsConstants.PLUGIN_NOT_DETECTED_KEY;
import static java.util.Objects.requireNonNull;

public class SimplePluginHelper implements PluginHelper {

    private final CallingBundleResolver callingBundleResolver;
    private final I18nResolver i18nResolver;
    private final PluginAccessor pluginAccessor;
    private final PluginMetadataManager pluginMetadataManager;

    public SimplePluginHelper(CallingBundleResolver callingBundleResolver, I18nResolver i18nResolver,
                              PluginAccessor pluginAccessor, PluginMetadataManager pluginMetadataManager) {
        this.callingBundleResolver = callingBundleResolver;
        this.i18nResolver = i18nResolver;
        this.pluginAccessor = pluginAccessor;
        this.pluginMetadataManager = pluginMetadataManager;
    }

    @Nonnull
    @Override
    public Optional<Bundle> getCallingBundle() {
        return callingBundleResolver.getCallingBundle();
    }

    @Nonnull
    @Override
    public PluginDetails getPluginDetails(@Nonnull String pluginKey, String pluginVersion) {
        return new PluginDetails(pluginKey, getPluginName(pluginKey), pluginVersion);
    }

    @Nonnull
    @Override
    public String getPluginName(@Nonnull String pluginKey) {
        requireNonNull(pluginKey, "pluginKey");

        if (PLUGIN_NOT_DETECTED_KEY.equals(pluginKey)) {
            return i18nResolver.getText("diagnostics.plugin.not.detected");
        }
        Plugin plugin = pluginAccessor.getPlugin(pluginKey);
        return plugin == null ? pluginKey : plugin.getName();
    }

    @Override
    public boolean isUserInstalled(Bundle bundle) {
        if (bundle != null) {
            if (bundle.getBundleId() == 0L) {
                // system bundle
                return false;
            }

            String pluginKey = OsgiHeaderUtil.getPluginKey(bundle);
            Plugin plugin = pluginAccessor.getPlugin(pluginKey);
            if (plugin != null) {
                return pluginMetadataManager.isUserInstalled(plugin);
            }
        }
        return false;
    }
}
