package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.AlertRequest;
import com.atlassian.diagnostics.ComponentMonitor;
import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.internal.concurrent.Gate;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

import static org.apache.commons.lang3.StringUtils.leftPad;

public abstract class InitializingMonitor {

    protected volatile ComponentMonitor monitor;

    private Map<Integer, Gate> issueGates = new ConcurrentHashMap<>();

    public abstract void init(MonitoringService monitoringService);

    protected void defineIssue(final String i18nPrefix, final int id, final Severity severity) {
        defineIssue(i18nPrefix, id, severity, null);
    }

    protected void defineIssue(final String i18nPrefix, final int id, final Severity severity, final Class<?> detailsClass) {
        final String keyPrefix = i18nPrefix + "." + leftPad(Integer.toString(id), 4, '0') + ".";
        monitor.defineIssue(id)
                .summaryI18nKey(keyPrefix + "summary")
                .descriptionI18nKey(keyPrefix + "description")
                .jsonMapper(detailsClass == null ? null : new JacksonJsonMapper<>(detailsClass))
                .severity(severity)
                .build();
        issueGates.put(id, new Gate(Duration.ofMinutes(15)));
    }

    protected void alert(final int issueId, final Consumer<AlertRequest.Builder> alertBuilder) {
        if (monitor != null && monitor.isEnabled()) {
            monitor.getIssue(issueId).ifPresent(issue -> {
                issueGates.get(issueId).ifAccessible(() -> {
                    final AlertRequest.Builder builder = new AlertRequest.Builder(issue);
                    alertBuilder.accept(builder);
                    monitor.alert(builder.build());
                });
            });
        }
    }
}

