package com.atlassian.diagnostics.internal.detail;

import com.atlassian.diagnostics.detail.ThreadDump;
import com.atlassian.diagnostics.detail.ThreadDumpProducer;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Produces thread dumps with a string representations of stack traces for the supplied threads. If the stack trace
 * of a thread is unavailable (per the contract of {@link Thread#getStackTrace()}), the {@link ThreadDump} is created
 * with an empty stack
 */
public class DefaultThreadDumpProducer implements ThreadDumpProducer {

    private static final int MAX_STACK_LENGTH = 128000;
    private static final ThreadMXBean threadMxBean = ManagementFactory.getThreadMXBean();

    @Nonnull
    @Override
    public List<ThreadDump> produce(@Nonnull Set<Thread> threads) {
        if (threads.isEmpty()) {
            return Collections.emptyList();
        }

        Map<Long, Thread> threadsById = threads.stream().collect(Collectors.toMap(Thread::getId, Function.identity()));
        long[] threadIds = threadsById.keySet().stream().mapToLong(Long::longValue).toArray();
        return Arrays.stream(threadMxBean.getThreadInfo(threadIds, MAX_STACK_LENGTH / 32)) // assume at least 32 chars per frame
                .filter(Objects::nonNull)
                .map(threadInfo -> new SimpleThreadDump(threadsById.get(threadInfo.getThreadId()),
                        toStackTraceString(Arrays.asList(threadInfo.getStackTrace()))))
                .collect(Collectors.toList());
    }

    private String toStackTraceString(List<StackTraceElement> elements) {
        if (elements.isEmpty()) {
            return null;
        }

        StringBuilder sb = new StringBuilder();
        elements.stream()
                .map(Object::toString)
                .forEach(elementLine -> {
                    if (sb.length() < MAX_STACK_LENGTH) {
                        sb.append(elementLine);
                        if (sb.length() < MAX_STACK_LENGTH) {
                            sb.append('\n');
                        }
                    }
                } );
        return StringUtils.trimToNull(sb.toString());
    }
}
