package com.atlassian.diagnostics.internal.dao;

import com.atlassian.diagnostics.Severity;
import com.google.common.base.MoreObjects;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.util.Objects;

import static com.atlassian.diagnostics.DiagnosticsConstants.PLUGIN_NOT_DETECTED_KEY;
import static java.util.Objects.requireNonNull;

/**
 * Metrics for a (issue, plugin, node) tuple
 */
public class AlertMetric {

    private final long count;
    private final String issueId;
    private final Severity issueSeverity;
    private final String nodeName;
    private final String pluginKey;
    private final String pluginVersion;

    public AlertMetric(@Nonnull String issueId, @Nonnull Severity severity, String pluginKey, String pluginVersion,
                       @Nonnull String nodeName, long count) {
        this.count = count;
        this.issueId = requireNonNull(issueId, "issueId");
        this.issueSeverity = requireNonNull(severity, "severity");
        this.nodeName = requireNonNull(nodeName, "nodeName");
        this.pluginKey = MoreObjects.firstNonNull(StringUtils.trimToNull(pluginKey), PLUGIN_NOT_DETECTED_KEY);
        this.pluginVersion = pluginVersion;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AlertMetric metric = (AlertMetric) o;
        return count == metric.count &&
                Objects.equals(issueId, metric.issueId) &&
                Objects.equals(nodeName, metric.nodeName) &&
                Objects.equals(pluginKey, metric.pluginKey) &&
                Objects.equals(pluginVersion, metric.pluginVersion);
    }

    public long getCount() {
        return count;
    }

    @Nonnull
    public String getIssueId() {
        return issueId;
    }

    @Nonnull
    public Severity getIssueSeverity() {
        return issueSeverity;
    }

    @Nonnull
    public String getNodeName() {
        return nodeName;
    }

    @Override
    public int hashCode() {
        return Objects.hash(issueId, nodeName, pluginKey, pluginVersion, count);
    }

    @Nonnull
    public String getPluginKey() {
        return pluginKey;
    }

    public String getPluginVersion() {
        return pluginVersion;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("issueId", issueId)
                .add("issueSeverity", issueSeverity)
                .add("nodeName", nodeName)
                .add("pluginKey", pluginKey)
                .add("pluginVersion", pluginVersion)
                .add("count", count)
                .toString();
    }
}
