package com.atlassian.diagnostics.internal;

import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.util.Comparator;
import java.util.Locale;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

public class IssueId implements Comparable<IssueId> {

    private static final Comparator<IssueId> COMPARATOR = Comparator.comparing(IssueId::toString);

    private final int code;
    private final String componentId;
    private final String idString;

    IssueId(String componentId, int code) {
        this.code = code;
        this.componentId = StringUtils.upperCase(componentId, Locale.ROOT);

        idString = this.componentId + "-" + StringUtils.leftPad(Integer.toString(code), 4, '0');
    }

    @Override
    public int compareTo(@Nonnull IssueId other) {
        return COMPARATOR.compare(this, other);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        IssueId issueId = (IssueId) o;
        return code == issueId.code && Objects.equals(componentId, issueId.componentId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(code, componentId);
    }

    @Override
    public String toString() {
        return idString;
    }

    @Nonnull
    public static IssueId valueOf(@Nonnull String id) {
        requireNonNull(id, "id");

        int separator = id.lastIndexOf('-');
        if (separator != -1 && separator < id.length() - 1) {
            String componentId = id.substring(0, separator);
            try {
                return new IssueId(componentId, Integer.parseInt(id.substring(separator + 1)));
            } catch (NumberFormatException e) {
                // ignore
            }
        }

        throw new IllegalArgumentException("Invalid issue ID: " + id);
    }

    int getCode() {
        return code;
    }

    @Nonnull
    String getComponentId() {
        return componentId;
    }
}
