package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertCount;
import com.atlassian.diagnostics.AlertCriteria;
import com.atlassian.diagnostics.AlertListener;
import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.AlertWithElisions;
import com.atlassian.diagnostics.Component;
import com.atlassian.diagnostics.ComponentMonitor;
import com.atlassian.diagnostics.DiagnosticsConfiguration;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.JsonMapper;
import com.atlassian.diagnostics.MonitorConfiguration;
import com.atlassian.diagnostics.PageCallback;
import com.atlassian.diagnostics.PageRequest;
import com.atlassian.diagnostics.PluginDetails;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.internal.dao.AlertEntity;
import com.atlassian.diagnostics.internal.dao.AlertEntityDao;
import com.atlassian.sal.api.lifecycle.LifecycleAware;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.permission.PermissionEnforcer;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.config.RunMode;
import com.atlassian.scheduler.config.Schedule;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import static com.atlassian.diagnostics.CallbackResult.CONTINUE;
import static com.atlassian.diagnostics.CallbackResult.DONE;
import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toSet;

public class DefaultMonitoringService implements LifecycleAware, InternalMonitoringService, IssueSupplier {

    private static final Logger log = LoggerFactory.getLogger(DefaultMonitoringService.class);

    static final JobRunnerKey JOB_RUNNER_KEY = JobRunnerKey.of(TruncateAlertsJobRunner.class.getName());
    static final JobId JOB_ID = JobId.of(TruncateAlertsJobRunner.class.getSimpleName());
    static final Duration WINDOW_SIZE = Duration.ofMinutes(1L);

    private static final int COLLECTOR_PAGE_FACTOR = 3;
    private static final MonitorConfiguration DEFAULT_MONITORING_CONFIGURATION = new MonitorConfiguration() {
        @Override
        public boolean isEnabled() {
            return true;
        }
    };

    private final DiagnosticsConfiguration configuration;
    private final AlertEntityDao dao;
    private final I18nResolver i18nResolver;
    private final JsonMapper jsonMapper;
    private final ConcurrentMap<String, InternalComponentMonitor> monitors;
    private final PermissionEnforcer permissionEnforcer;
    private final ConcurrentMap<String, Component> placeholderComponents;
    private final ConcurrentMap<IssueId, Issue> placeholderIssues;
    private final PluginHelper pluginHelper;
    private final AlertPublisher publisher;
    private final SchedulerService schedulerService;
    private final TransactionTemplate transactionTemplate;

    public DefaultMonitoringService(final DiagnosticsConfiguration configuration,
                                    final AlertEntityDao dao,
                                    final I18nResolver i18nResolver,
                                    final JsonMapper jsonMapper,
                                    final PermissionEnforcer permissionEnforcer,
                                    final PluginHelper pluginHelper,
                                    final AlertPublisher publisher,
                                    final SchedulerService schedulerService,
                                    final TransactionTemplate transactionTemplate) {
        this.configuration = configuration;
        this.dao = dao;
        this.i18nResolver = i18nResolver;
        this.jsonMapper = jsonMapper;
        this.permissionEnforcer = permissionEnforcer;
        this.pluginHelper = pluginHelper;
        this.publisher = publisher;
        this.schedulerService = schedulerService;
        this.transactionTemplate = transactionTemplate;

        monitors = new ConcurrentHashMap<>();
        placeholderComponents = new ConcurrentHashMap<>();
        placeholderIssues = new ConcurrentHashMap<>();
    }

    @Nonnull
    @Override
    public ComponentMonitor createMonitor(@Nonnull final String componentId, @Nonnull final String componentNameI18nKey) {
        return createMonitor(componentId, componentNameI18nKey, DEFAULT_MONITORING_CONFIGURATION);
    }

    @Nonnull
    @Override
    public ComponentMonitor createMonitor(@Nonnull final String componentId, @Nonnull final String componentNameI18nKey, @Nonnull final MonitorConfiguration monitorConfiguration) {
        requireNonNull(componentId, "componentId");
        requireNonNull(componentNameI18nKey, "componentNameI18nKey");
        requireNonNull(monitorConfiguration, "monitorConfiguration");

        String upperComponentId = componentId.toUpperCase();
        return monitors.computeIfAbsent(upperComponentId,
                id -> internalCreateMonitor(upperComponentId, componentNameI18nKey, monitorConfiguration));
    }

    private InternalComponentMonitor internalCreateMonitor(@Nonnull final String componentId, @Nonnull final String nameI18nKey, @Nonnull final MonitorConfiguration monitorConfiguration) {
        return new DefaultComponentMonitor(new SimpleComponent(i18nResolver, componentId, nameI18nKey),
                configuration, monitorConfiguration, i18nResolver, jsonMapper, publisher);
    }

    @Override
    public boolean destroyMonitor(@Nonnull String componentId) {
        InternalComponentMonitor componentMonitor = monitors.remove(componentId);
        if (componentMonitor != null) {
            componentMonitor.destroy();
        }
        return componentMonitor != null;
    }

    @Nonnull
    @Override
    public Set<Component> findAllComponents() {
        ImmutableSet.Builder<Component> builder = ImmutableSet.builder();
        Set<String> componentsIdsWithAlerts = transactionTemplate.execute(dao::findAllComponentIds).stream()
                .map(value -> StringUtils.upperCase(value, Locale.ROOT))
                .collect(toSet());

        // add all currently defined components
        monitors.values().stream()
                .map(ComponentMonitor::getComponent)
                .peek(component -> componentsIdsWithAlerts.remove(component.getId()))
                .forEach(builder::add);

        // add placeholders for all components with alerts from the database that are not currently defined
        componentsIdsWithAlerts.forEach(id -> builder.add(getPlaceholderComponent(id)));
        return builder.build();
    }

    @Nonnull
    @Override
    public Set<Issue> findAllIssues() {
        ImmutableSet.Builder<Issue> builder = ImmutableSet.builder();
        Map<String, Severity> issueIdsWithAlerts = new HashMap<>(transactionTemplate.execute(dao::findAllIssueIds));

        monitors.values().stream()
                .flatMap(monitor -> monitor.getIssues().stream())
                .peek(issue -> issueIdsWithAlerts.remove(issue.getId()))
                .forEach(builder::add);
        issueIdsWithAlerts.forEach((id, severity) -> builder.add(getIssueOrPlaceholder(id, severity)));
        return builder.build();
    }

    @Nonnull
    @Override
    public Set<String> findAllNodesWithAlerts() {
        return transactionTemplate.execute(dao::findAllNodeNames);
    }

    @Nonnull
    @Override
    public Set<PluginDetails> findAllPluginsWithAlerts() {
        Set<String> pluginKeys = transactionTemplate.execute(dao::findAllPluginKeys);
        Set<PluginDetails> pluginDetails = pluginKeys.stream().map(key -> new PluginDetails(key, pluginHelper.getPluginName(key), null))
                .collect(toSet());
        log.info("pluginKeys: [{}], pluginDetails: [{}]", pluginDetails);
        return pluginDetails;
    }

    @Nonnull
    @Override
    public Issue getIssue(@Nonnull String issueId, @Nonnull Severity severity) {
        return getIssueOrPlaceholder(issueId, severity);
    }

    @Nonnull
    @Override
    public Optional<ComponentMonitor> getMonitor(@Nonnull String componentId) {
        requireNonNull(componentId, "componentId");

        return ofNullable(monitors.get(componentId.toUpperCase()));
    }

    public <T> T internalStreamAlertCounts(@Nonnull AlertCriteria criteria,
                                           @Nonnull PageCallback<? super AlertCount, T> callback,
                                           @Nonnull PageRequest pageRequest) {
        requireNonNull(callback, "callback");
        requireNonNull(criteria, "criteria");
        requireNonNull(pageRequest, "pageRequest");

        MutableBoolean done = new MutableBoolean(false);
        MutableInt emittedRow = new MutableInt(0);
        MutableInt entityRow = new MutableInt(0);
        int start = pageRequest.getStart();
        int limit = pageRequest.getLimit();
        int end = start + limit;
        AlertCountCollector collector = new AlertCountCollector(this, pluginHelper);
        T result;
        int daoPageSize = limit * COLLECTOR_PAGE_FACTOR;
        try {
            callback.onStart(pageRequest);
            while (done.isFalse() && emittedRow.getValue() <= end) {
                transactionTemplate.execute(() -> {
                    MutableInt pageSize = new MutableInt(0);
                    dao.streamMetrics(criteria, entity -> {
                        if (pageSize.incrementAndGet() > daoPageSize) {
                            // this is the extra item that signals that there is a next page. Don't send it to the
                            // collector because it'll be re-retrieved on the next page.
                            return CONTINUE;
                        }
                        entityRow.increment();
                        AlertCount alertCount = collector.onRow(entity);
                        if (alertCount != null) {
                            int row = emittedRow.getAndIncrement();
                            if (row >= start && row < end) {
                                if (callback.onItem(alertCount) == DONE) {
                                    done.setTrue();
                                }
                            }
                        }
                        done.setValue(done.isTrue() || emittedRow.getValue() > end);
                        return done.isTrue() ? DONE : CONTINUE;
                    }, PageRequest.of(entityRow.getValue(), daoPageSize));
                    done.setValue(done.isTrue() || pageSize.getValue() <= daoPageSize);
                    return null;
                });
            }
        } finally {
            AlertCount finalItem = collector.onEnd();
            if (finalItem != null) {
                int row = emittedRow.getAndIncrement();
                if (row >= start && row < end) {
                    // full page, or callback signaled DONE
                    callback.onItem(finalItem);
                }
            }

            PageRequest prevRequest = start == 0 ? null : PageRequest.of(Math.max(0, start - limit), limit);
            PageRequest nextRequest = emittedRow.getValue() <= end ? null : PageRequest.of(end, limit);
            int size = Math.min(limit, emittedRow.getValue() - start);
            result = callback.onEnd(new SimplePageSummary(prevRequest, nextRequest, size));
        }
        return result;
    }

    @Override
    public boolean isEnabled() {
        return configuration.isEnabled();
    }

    @Override
    public void onStart() {
        long intervalInMillis = configuration.getAlertTruncationInterval().toMillis();
        long firstRunTime = System.currentTimeMillis() + intervalInMillis;
        schedulerService.registerJobRunner(JOB_RUNNER_KEY, new TruncateAlertsJobRunner());
        try {
            schedulerService.scheduleJob(JOB_ID, JobConfig.forJobRunnerKey(JOB_RUNNER_KEY)
                    .withSchedule(Schedule.forInterval(intervalInMillis, new Date(firstRunTime)))
                    .withRunMode(RunMode.RUN_ONCE_PER_CLUSTER));
        } catch (SchedulerServiceException e) {
            log.warn("Failed to schedule periodic alert truncation", e);
        }
    }

    @Override
    public void onStop() {
        schedulerService.unregisterJobRunner(JOB_RUNNER_KEY);
    }

    @Override
    public <T> T streamAlerts(@Nonnull AlertCriteria criteria, @Nonnull PageCallback<? super Alert, T> callback,
                              @Nonnull PageRequest pageRequest) {
        requireNonNull(callback, "callback");
        requireNonNull(criteria, "criteria");
        requireNonNull(pageRequest, "pageRequest");

        permissionEnforcer.enforceSystemAdmin();

        return transactionTemplate.execute(() -> {
            T result;
            callback.onStart(pageRequest);
            MutableInt count = new MutableInt(0);
            int limit = pageRequest.getLimit();
            try {
                dao.streamAll(criteria, entity -> {
                    if (count.incrementAndGet() > limit) {
                        // this is limit + 1 item - we have a full page + an extra item to signal that there's a next
                        // page. 'limit' items have been provided to the callback. All done!
                        return DONE;
                    }
                    return callback.onItem(toAlert(entity));
                }, pageRequest);
            } finally {
                result = callback.onEnd(new SimplePageSummary(pageRequest, count.getValue()));
            }
            return result;
        });
    }

    @Override
    public <T> T streamAlertCounts(@Nonnull AlertCriteria criteria,
                                   @Nonnull PageCallback<? super AlertCount, T> callback,
                                   @Nonnull PageRequest pageRequest) {
        permissionEnforcer.enforceSystemAdmin();

        return internalStreamAlertCounts(criteria, callback, pageRequest);
    }

    @Override
    public <T> T streamAlertsWithElisions(@Nonnull AlertCriteria criteria,
                                          @Nonnull PageCallback<? super AlertWithElisions, T> callback,
                                          @Nonnull PageRequest pageRequest) {
        requireNonNull(callback, "callback");
        requireNonNull(criteria, "criteria");
        requireNonNull(pageRequest, "pageRequest");

        permissionEnforcer.enforceSystemAdmin();

        return transactionTemplate.execute(() -> {
            AlertWithElisionsCollector collector = new AlertWithElisionsCollector(this, pageRequest, WINDOW_SIZE);

            // step 1: scan through the alerts without loading the (potentially large) alert details to determine
            // the best representative for each alert group per window, and the number of alerts we'll elide
            dao.streamMinimalAlerts(criteria, alert -> {
                collector.add(alert);
                return collector.hasCompletePage() ? DONE : CONTINUE;
            }, PageRequest.ofSize(Integer.MAX_VALUE - 1));
            collector.onEndAlertScan();

            // step 2: fully load the best representative for each value that is to be returned and emit it to the
            // provided callback
            MutableInt count = new MutableInt(0);
            int limit = pageRequest.getLimit();
            T result;
            callback.onStart(pageRequest);
            try {
                MutableBoolean done = new MutableBoolean(false);
                dao.streamByIds(collector.getAlertIdsToLoad(), entity -> {
                    for (AlertWithElisions alertWithElisions : collector.resolveCandidate(entity)) {
                        if (count.incrementAndGet() > limit) {
                            // this is limit + 1 item - we have a full page + an extra item to signal that there's a next
                            // page. 'limit' items have been provided to the callback. All done!
                            return DONE;
                        }

                        if (callback.onItem(alertWithElisions) == DONE) {
                            done.setTrue();
                            return DONE;
                        }
                    }

                    return CONTINUE;
                });
                if (!done.isTrue()) {
                    for (AlertWithElisions alert : collector.onEndAlertResolution()) {
                        if (count.incrementAndGet() > limit || callback.onItem(alert) == DONE) {
                            break;
                        }
                    }
                }
            } finally {
                result = callback.onEnd(new SimplePageSummary(pageRequest, count.getValue()));
            }
            return result;
        });
    }

    @Nonnull
    @Override
    public String subscribe(@Nonnull AlertListener listener) {
        return publisher.subscribe(listener);
    }

    @Override
    public boolean unsubscribe(@Nonnull String subscriptionId) {
        return publisher.unsubscribe(subscriptionId);
    }

    private Issue getIssueOrPlaceholder(String issueId, Severity severity) {
        return getIssueOrPlaceholder(IssueId.valueOf(issueId), severity);
    }

    private Issue getIssueOrPlaceholder(IssueId issueId, Severity severity) {
        return getMonitor(issueId.getComponentId())
                .flatMap(monitor -> monitor.getIssue(issueId.getCode()))
                .orElseGet(() -> getPlaceHolderIssue(issueId, severity));
    }

    private Component getPlaceholderComponent(String componentId) {
        String upperComponentId = componentId.toUpperCase();
        return placeholderComponents.computeIfAbsent(upperComponentId,
                id -> new PlaceholderComponent(upperComponentId));
    }

    private Issue getPlaceHolderIssue(IssueId id, Severity severity) {
        Component component = getMonitor(id.getComponentId())
                .map(ComponentMonitor::getComponent)
                .orElseGet(() -> getPlaceholderComponent(id.getComponentId()));
        return placeholderIssues.computeIfAbsent(id,
                issueId -> new PlaceholderIssue(i18nResolver, component, id, severity, jsonMapper));
    }

    private Alert toAlert(AlertEntity entity) {
        Issue issue = getIssueOrPlaceholder(entity.getIssueId(), entity.getIssueSeverity());
        return new SimpleAlert.Builder(issue, entity.getNodeName())
                .id(entity.getId())
                .detailsAsJson(entity.getDetailsJson())
                .timestamp(entity.getTimestamp())
                .trigger(new AlertTrigger.Builder()
                        .plugin(entity.getTriggerPluginKey(), entity.getTriggerPluginVersion())
                        .module(entity.getTriggerModule())
                        .build())
                .build();
    }

    private void truncateAlerts() {
        transactionTemplate.execute(() -> {
            dao.deleteAll(AlertCriteria.builder()
                    .until(Instant.now().minus(configuration.getAlertRetentionPeriod()))
                    .build());
            return null;
        });
    }

    private static class PlaceholderComponent implements Component {

        private final String id;

        PlaceholderComponent(String id) {
            this.id = requireNonNull(id, "id").toUpperCase(Locale.ROOT);
        }

        @Nonnull
        @Override
        public String getId() {
            return id;
        }

        @Nonnull
        @Override
        public String getName() {
            return id;
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("id", id)
                    .toString();
        }
    }

    private static class PlaceholderIssue extends SimpleIssue {

        PlaceholderIssue(I18nResolver i18nResolver, Component component, IssueId id, Severity severity,
                         JsonMapper jsonMapper) {
            super(i18nResolver, component, id, "diagnostics.unknown.issue", "diagnostics.unknown.issue",
                    firstNonNull(severity, Severity.WARNING), jsonMapper);
        }

        @Nonnull
        @Override
        public String getSummary() {
            return getId();
        }

        @Nonnull
        @Override
        public String getDescription() {
            return i18nResolver.getText("diagnostics.unknown.issue", getId());
        }
    }

    private class TruncateAlertsJobRunner implements JobRunner {

        @Override
        public JobRunnerResponse runJob(@Nonnull JobRunnerRequest jobRunnerRequest) {
            truncateAlerts();
            return JobRunnerResponse.success();
        }
    }
}
