package com.atlassian.diagnostics.internal.jmx;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertListener;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.internal.PluginHelper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import java.lang.management.ManagementFactory;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

public class JmxReportingAlertListener implements AlertListener {

    private static final Logger log = LoggerFactory.getLogger(JmxReportingAlertListener.class);

    private final Alerts alerts;
    private final ConcurrentMap<String, IssueAlerts> alertsByIssue;
    private final ConcurrentMap<String, PluginAlerts> alertsByPlugin;
    private final PluginHelper pluginHelper;

    public JmxReportingAlertListener(PluginHelper pluginHelper) {
        this.pluginHelper = pluginHelper;

        alerts = new Alerts();
        alertsByIssue = new ConcurrentHashMap<>();
        alertsByPlugin = new ConcurrentHashMap<>();
        registerJmxBean("type=Alerts,name=Total", alerts);
    }

    @Override
    public void onAlert(@Nonnull Alert alert) {
        alerts.onAlert(alert);

        alertsByPlugin.computeIfAbsent(alert.getTrigger().getPluginKey(), this::createPluginAlerts)
                .onAlert(alert);
        alertsByIssue.computeIfAbsent(alert.getIssue().getId(), id -> createIssueAlerts(alert.getIssue()))
                .onAlert(alert);
    }

    private IssueAlerts createIssueAlerts(Issue issue) {
        IssueAlerts result = new IssueAlerts(issue);
        registerJmxBean("type=Alerts,Category=Issue,name=" + issue.getId(), result);
        return result;
    }

    private PluginAlerts createPluginAlerts(String pluginKey) {
        PluginAlerts result;
        if ("System".equalsIgnoreCase(pluginKey)) {
            result = new PluginAlerts("System");
        } else {
            result = new PluginAlerts(pluginHelper.getPluginName(pluginKey));
            registerJmxBean("type=Alerts,Category=Plugin,name=" + maybeQuote(pluginKey), result);
        }
        return result;
    }

    private String maybeQuote(String value) {
        if (StringUtils.containsAny(value, ',')) {
            return ObjectName.quote(value);
        }
        return value;
    }

    private void registerJmxBean(String objectNameProperties, Object alerts) {
        try {
            MBeanServer mbeanServer = ManagementFactory.getPlatformMBeanServer();
            ObjectName mxbeanName = new ObjectName("com.atlassian.diagnostics:" + objectNameProperties);
            mbeanServer.registerMBean(alerts, mxbeanName);
        } catch (RuntimeException | JMException e) {
            log.warn("Failed to register Alerts JMX bean", e);
        }
    }
}
