package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertBuilder;
import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.Issue;
import org.slf4j.Logger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.slf4j.LoggerFactory.getLogger;

public class SimpleAlert implements Alert {

    private static final Logger log = getLogger(SimpleAlert.class);

    private final Object details;
    private final long id;
    private final Issue issue;
    private final String nodeName;
    private final long timestamp;
    private final AlertTrigger trigger;

    SimpleAlert(AbstractBuilder<?> builder) {
        id = builder.id;
        details = builder.details;
        issue = builder.issue;
        nodeName = builder.nodeName;
        timestamp = builder.timestamp;
        trigger = builder.trigger == null ? new AlertTrigger.Builder().build() : builder.trigger;
    }

    @Override
    public long getId() {
        return id;
    }

    @Nonnull
    @Override
    public Issue getIssue() {
        return issue;
    }

    @Nonnull
    @Override
    public Optional<Object> getDetails() {
        return ofNullable(details);
    }

    @Nonnull
    @Override
    public String getNodeName() {
        return nodeName;
    }

    @Nonnull
    @Override
    public Instant getTimestamp() {
        return Instant.ofEpochMilli(timestamp);
    }

    @Nonnull
    @Override
    public AlertTrigger getTrigger() {
        return trigger;
    }

    protected abstract static class AbstractBuilder<B extends AbstractBuilder<B>> implements AlertBuilder {

        private final Issue issue;
        private final String nodeName;

        private Object details;
        private long id;
        private long timestamp;
        private AlertTrigger trigger;

        protected AbstractBuilder(@Nonnull Issue issue, @Nonnull String nodeName) {
            this.issue = requireNonNull(issue, "issue");
            this.nodeName = requireNonNull(nodeName, "nodeName");
        }

        @Nonnull
        @Override
        public B details(Object value) {
            details = value;
            return self();
        }

        @Nonnull
        @Override
        public B detailsAsJson(String value) {
            try {
                details = issue.getJsonMapper().parseJson(value);
            } catch (Exception e) {
                log.warn("Failed to parse json for alert details relating to issue '{}' {}", issue.getId(),
                        log.isDebugEnabled() ? value : "");
                details = null;
            }
            return self();
        }

        @Nonnull
        public B id(long value) {
            id = value;
            return self();
        }

        @Nonnull
        public B timestamp(@Nonnull Instant value) {
            timestamp = requireNonNull(value, "timestamp").toEpochMilli();
            return self();
        }

        @Nonnull
        public B trigger(@Nullable AlertTrigger value) {
            trigger = value;
            return self();
        }

        @Nonnull
        protected abstract B self();
    }

    public static class Builder extends AbstractBuilder<Builder> {


        public Builder(@Nonnull Issue issue, @Nonnull String nodeName) {
            super(issue, nodeName);
        }

        @Nonnull
        public SimpleAlert build() {
            return new SimpleAlert(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
