package com.atlassian.diagnostics.internal;

import com.atlassian.net.NetworkUtils;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.time.Duration;
import java.util.Enumeration;
import java.util.Set;

public class DefaultDiagnosticsConfiguration implements DiagnosticsConfiguration {

    private static final Set<String> LOCALHOST_IPS = ImmutableSet.of("127.0.0.1", "0:0:0:0:0:0:0:1", "::1");

    private volatile String nodeName;

    @Nonnull
    @Override
    public String getNodeName() {
        if (nodeName == null) {
            try {
                InetAddress address = null;
                Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
                while (interfaces.hasMoreElements()) {
                    address = getBestAddress(address, getBestAddress(interfaces.nextElement()));
                }
                String hostName = NetworkUtils.getLocalHostName();
                if ("localhost".equalsIgnoreCase(hostName) && address != null) {
                    // if we can't resolve the hostname, use the IP address (if it's something else than 127.0.0.1)
                    nodeName = LOCALHOST_IPS.contains(address.getHostAddress()) ? hostName : address.getHostAddress();
                }
                nodeName = hostName;
            } catch (SocketException | UnknownHostException e) {
                nodeName = "unknown";
            }
        }
        return nodeName;
    }

    @Nonnull
    @Override
    public Duration getAlertRetentionPeriod() {
        return Duration.ofDays(30L);
    }

    @Nonnull
    @Override
    public Duration getAlertTruncationInterval() {
        return Duration.ofDays(1L);
    }

    @Override
    public boolean isEnabled() {
        return true;
    }

    private InetAddress getBestAddress(NetworkInterface networkInterface) {
        InetAddress best = null;
        Enumeration<InetAddress> addresses = networkInterface.getInetAddresses();
        while (addresses.hasMoreElements()) {
            best = getBestAddress(best, addresses.nextElement());
        }
        return best;
    }

    private InetAddress getBestAddress(InetAddress currentBest, InetAddress candidate) {
        if (currentBest == null) {
            return candidate;
        }
        if (isLocal(currentBest) && !isLocal(candidate)) {
            return candidate;
        }
        if (!(currentBest instanceof Inet4Address) && candidate instanceof Inet4Address) {
            // prefer IPv4 addresses because they're easier to read
            return candidate;
        }

        return currentBest; // stick to the first candidate
    }

    private boolean isLocal(InetAddress address) {
        return address != null &&
                (address.isSiteLocalAddress() || address.isLinkLocalAddress() || address.isLoopbackAddress());
    }
}
