package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertRequest;
import com.atlassian.diagnostics.Component;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.IssueBuilder;
import com.atlassian.diagnostics.JsonMapper;
import com.atlassian.diagnostics.Severity;
import com.atlassian.sal.api.message.I18nResolver;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.slf4j.LoggerFactory.getLogger;

class DefaultComponentMonitor implements InternalComponentMonitor {
    private static final Logger log = getLogger(DefaultComponentMonitor.class);

    private final Component component;
    private final DiagnosticsConfiguration configuration;
    private final JsonMapper defaultJsonMapper;
    private final I18nResolver i18nResolver;
    private final ConcurrentMap<Integer, Issue> issues;
    private final AlertPublisher publisher;
    private final AtomicBoolean destroyed = new AtomicBoolean();

    @VisibleForTesting
    DefaultComponentMonitor(Component component, DiagnosticsConfiguration configuration,
                            I18nResolver i18nResolver, JsonMapper defaultJsonMapper, AlertPublisher publisher) {
        this.component = component;
        this.configuration = configuration;
        this.i18nResolver = i18nResolver;
        this.defaultJsonMapper = defaultJsonMapper;
        this.publisher = publisher;

        issues = new ConcurrentHashMap<>();
    }

    @Override
    public void alert(@Nonnull AlertRequest request) {
        checkMonitorState();
        if (!configuration.isEnabled()) {
            return;
        }

        requireNonNull(request, "request");

        Issue issue = request.getIssue();
        if (!component.equals(issue.getComponent())) {
            throw new IllegalArgumentException("Issue " + issue.getId() + " is unknown for component " +
                    component.getId());
        }
        // build the alert
        Alert alert = new SimpleAlert.Builder(issue, configuration.getNodeName())
                .details(request.getDetailsSupplier().map(Supplier::get).orElse(null))
                .trigger(request.getTrigger().orElse(null))
                .timestamp(request.getTimestamp())
                .build();

        // publish the alert
        publisher.publish(alert);
    }

    @Nonnull
    @Override
    public IssueBuilder defineIssue(int issueId) {
        checkMonitorState();
        if (issueId <= 0 || issueId > 9999) {
            throw new IllegalArgumentException("issueId must be greater than 0 and less than 10000");
        }
        if (issues.containsKey(issueId)) {
            throw new IllegalStateException("Issue " + issueId + " is already defined");
        }

        return new ComponentIssueBuilder(issueId).jsonMapper(defaultJsonMapper);
    }

    @Nonnull
    @Override
    public Component getComponent() {
        return component;
    }

    @Nonnull
    @Override
    public Optional<Issue> getIssue(int issueId) {
        return ofNullable(issues.get(issueId));
    }

    @Nonnull
    @Override
    public List<Issue> getIssues() {
        return ImmutableList.copyOf(issues.values());
    }

    @Override
    public boolean isEnabled() {
        checkMonitorState();
        return configuration.isEnabled();
    }

    private void checkMonitorState() {
        if (destroyed.get()) {
            log.error("ComponentMonitor '{}' has been destroyed", component.getId());
        }
    }

    public void destroy() {
        checkMonitorState();
        destroyed.set(true);
    }

    private class ComponentIssueBuilder implements IssueBuilder {

        private final int id;

        private String descriptionI18nKey;
        private JsonMapper jsonMapper;
        private String summaryI18nKey;
        private Severity severity;

        ComponentIssueBuilder(int id) {
            checkMonitorState();
            this.id = id;

            severity = Severity.INFO;
        }

        @Nonnull
        @Override
        public Issue build() {
            Issue issue = new SimpleIssue(i18nResolver, component, new IssueId(component.getId(), id), summaryI18nKey,
                    descriptionI18nKey, severity, firstNonNull(jsonMapper, defaultJsonMapper));

            Issue existing = issues.putIfAbsent(id, issue);
            if (existing != null) {
                throw new IllegalStateException("Issue " + id + " has already been defined");
            }
            return issue;
        }

        @Nonnull
        @Override
        public ComponentIssueBuilder descriptionI18nKey(@Nonnull String value) {
            descriptionI18nKey = requireNonNull(value, "descriptionI18nKey");
            return this;
        }

        @Nonnull
        @Override
        public ComponentIssueBuilder jsonMapper(JsonMapper value) {
            jsonMapper = value;
            return this;
        }

        @Nonnull
        @Override
        public ComponentIssueBuilder severity(@Nonnull Severity value) {
            severity = requireNonNull(value, "severity");
            return this;
        }

        @Nonnull
        @Override
        public ComponentIssueBuilder summaryI18nKey(@Nonnull String value) {
            summaryI18nKey = requireNonNull(value, "summaryI18nKey");
            return this;
        }
    }
}
