package com.atlassian.diagnostics;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Optional;

/**
 * Monitor that can be used to {@link #alert raise} {@link Alert alerts} when an {@link Issue issue} is detected
 */
public interface ComponentMonitor {

    /**
     * Raises a new alert for the {@link #getComponent() component}.
     *
     * @param request alert details, including the identified {@link Issue}
     */
    void alert(@Nonnull AlertRequest request);

    /**
     * Defines a new {@link Issue issue} for the {@link #getComponent() component}, which be used when
     * {@link #alert raising alerts}.
     *
     * @param issueId the numeric issue ID, which must be in the 1-9999 range
     * @return a builder for the issue
     */
    @Nonnull
    IssueBuilder defineIssue(int issueId);

    /**
     * @return the component for which this is a monitor
     */
    @Nonnull
    Component getComponent();

    /**
     * @param issueId the component-specific ID of the issue to retrieve
     * @return the issue, if defined for this monitor, otherwise {@link Optional#empty()}
     */
    @Nonnull
    Optional<Issue> getIssue(int issueId);

    /**
     * @return the issues configured for this monitor
     */
    @Nonnull
    List<Issue> getIssues();

    /**
     * @return {@code true} if the monitoring feature is enabled, otherwise {@code false}
     */
    boolean isEnabled();
}
