package com.atlassian.diagnostics;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;
import java.util.Optional;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Request to raise an alert
 */
public class AlertRequest {

    private final Supplier<Object> detailsSupplier;
    private final Issue issue;
    private final Instant timestamp;
    private final AlertTrigger trigger;

    private AlertRequest(Builder builder) {
        issue = builder.issue;
        detailsSupplier = builder.detailsSupplier;
        timestamp = builder.timestamp;
        trigger = builder.trigger;
    }

    /**
     * @param issue the issue to which the alert applies
     * @return a new builder
     */
    @Nonnull
    public static AlertRequest.Builder builder(@Nonnull Issue issue) {
        return new AlertRequest.Builder(issue);
    }

    /**
     * @return the supplier to be used to generate alert details if required
     */
    @Nonnull
    public Optional<Supplier<Object>> getDetailsSupplier() {
        return ofNullable(detailsSupplier);
    }

    /**
     * @return the issue to which the alert applies
     */
    @Nonnull
    public Issue getIssue() {
        return issue;
    }

    /**
     * @return the timestamp of the alert
     */
    @Nonnull
    public Instant getTimestamp() {
        return timestamp;
    }

    /**
     * @return the trigger for the detected {@link Issue}
     */
    @Nonnull
    public Optional<AlertTrigger> getTrigger() {
        return ofNullable(trigger);
    }

    public static class Builder {

        private final Issue issue;

        private Supplier<Object> detailsSupplier;
        private Instant timestamp;
        private AlertTrigger trigger;

        public Builder(@Nonnull Issue issue) {
            this.issue = requireNonNull(issue, "issue");

            timestamp = Instant.now();
        }

        @Nonnull
        public AlertRequest build() {
            return new AlertRequest(this);
        }

        @Nonnull
        public Builder details(@Nullable Supplier<Object> value) {
            detailsSupplier = value;
            return this;
        }

        @Nonnull
        public Builder timestamp(@Nullable Instant timestamp) {
            if (timestamp != null) {
                this.timestamp = timestamp;
            }
            return this;
        }

        @Nonnull
        public Builder trigger(@Nullable AlertTrigger value) {
            trigger = value;
            return this;
        }
    }
}
