package com.atlassian.diagnostics;

import javax.annotation.Nonnull;
import java.util.Optional;
import java.util.Set;

public interface MonitoringService {

    /**
     * Creates a new monitor for a component with the provided ID and name, or returns the existing monitor if it
     * has already been created.
     *
     * @param componentId the {@link Component#getId() component ID}
     * @param componentNameI18nKey the i18n key of the component name
     * @return the monitor for the component with ID {@code componentId}
     * @deprecated use {@link MonitoringService#createMonitor(String, String, MonitorConfiguration)}
     */
    @Deprecated
    @Nonnull
    ComponentMonitor createMonitor(@Nonnull String componentId, @Nonnull String componentNameI18nKey);

    /**
     * Creates a new monitor for a component with the provided ID and name, or returns the existing monitor if it
     * has already been created.
     *
     * @param componentId the {@link Component#getId() component ID}
     * @param componentNameI18nKey the i18n key of the component name
     * @return the monitor for the component with ID {@code componentId}
     */
    @Nonnull
    ComponentMonitor createMonitor(@Nonnull String componentId, @Nonnull String componentNameI18nKey, @Nonnull MonitorConfiguration monitorConfiguration);

    /**
     * Destroys the monitor.
     *
     * @param componentId the {@link Component#getId() component ID}
     * @return true if the monitor was found and destroyed and false if the monitor was not found
     * @since 1.1
     */
    boolean destroyMonitor(@Nonnull String componentId);

    /**
     * @return a set of all {@link Component components} that are currently defined in the system or are no longer
     *         defined, but for which alerts have been detected recently (30 days by default)
     */
    @Nonnull
    Set<Component> findAllComponents();

    /**
     * @return a set of all {@link Issue issues} that are currently defined in the system or are no longer
     *         defined, but for which alerts have been detected recently (30 days by default)
     */
    @Nonnull
    Set<Issue> findAllIssues();

    /**
     * @return all nodes for which alerts have been detected recently (30 days by default)
     */
    @Nonnull
    Set<String> findAllNodesWithAlerts();

    /**
     * @return all {@link PluginDetails plugins} for which alerts have been detected recently (30 days by default)
     */
    @Nonnull
    Set<PluginDetails> findAllPluginsWithAlerts();

    /**
     * @param componentId the {@link Component#getId() component ID}
     * @return the monitor for the component with ID {@code componentId}, or empty if no such monitor has been
     *          created
     */
    @Nonnull
    Optional<ComponentMonitor> getMonitor(@Nonnull String componentId);

    /**
     * @return {@code true} if the monitoring feature is enabled, otherwise {@code false}
     */
    boolean isEnabled();

    /**
     * Subscribes the provided {@code listener} to all alerts
     *
     * @param listener the listener
     * @return the subscription ID to be used in {@link #unsubscribe}
     */
    @Nonnull
    String subscribe(@Nonnull AlertListener listener);

    /**
     * @param criteria describes which alerts to retrieve
     * @param callback the callback to send the retrieved alerts to
     * @param pageRequest specifies page limits
     * @return the callback result
     */
    <T> T streamAlerts(@Nonnull AlertCriteria criteria, @Nonnull PageCallback<? super Alert, T> callback,
                       @Nonnull PageRequest pageRequest);


    <T> T streamAlertCounts(@Nonnull AlertCriteria criteria,
                            @Nonnull PageCallback<? super AlertCount, T> callback,
                            @Nonnull PageRequest pageRequest);
    /**
     * @param criteria describes which alerts to retrieve
     * @param callback the callback to send the retrieved alerts to
     * @param pageRequest specifies page limits
     * @return the callback result
     */
    <T> T streamAlertsWithElisions(@Nonnull AlertCriteria criteria,
                                   @Nonnull PageCallback<? super AlertWithElisions, T> callback,
                                   @Nonnull PageRequest pageRequest);

    /**
     * Unsubscribes the {@link AlertListener} that had previously {@link #subscribe subscribed}.
     *
     * @param subscriptionId the subscription ID returned by {@link #subscribe}
     * @return {@code true} if the listener was unsubscribed, {@code false} if no subscription was found for
     *          {@code subscriptionId}
     */
    boolean unsubscribe(@Nonnull String subscriptionId);
}
