package com.atlassian.diagnostics;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.upperCase;

/**
 * The criteria for including an {@link Alert} in an operation (e.g. streaming a sequence of {@link Alert alerts}).
 */
public class AlertCriteria {

    /**
     * The maximum number of entries that can be defined in {@link #getComponentIds()}, {@link #getIssueIds()},
     * {@link #getNodeNames()} or {@link #getPluginKeys()}. Specifying more than the maximum amount results in an
     * {@link IllegalArgumentException} being thrown from {@link Builder#build()}.
     */
    private static final int MAX_VALUES_IN_FIELD = 150;
    private static final Function<String, String> TO_UPPER = val -> upperCase(val, Locale.ROOT);

    private final Set<String> componentIds;
    private final Set<String> issueIds;
    private final Set<String> nodeNames;
    private final Set<String> pluginKeys;
    private final Set<Severity> severities;
    private final Instant since;
    private final Instant until;

    private AlertCriteria(Builder builder) {
        componentIds = checkMaxSizeNotExceeded(builder.componentIds.build(), "component IDs");
        issueIds = checkMaxSizeNotExceeded(builder.issueIds.build(), "issue IDs");
        nodeNames = checkMaxSizeNotExceeded(builder.nodes.build(), "node names");
        pluginKeys = checkMaxSizeNotExceeded(builder.pluginKeys.build(), "plugin keys");
        severities = checkMaxSizeNotExceeded(builder.severities.build(), "severities");
        since = builder.since;
        until = builder.until;
    }

    @Nonnull
    public static Builder builder() {
        return new AlertCriteria.Builder();
    }

    @Nonnull
    public Set<String> getComponentIds() {
        return componentIds;
    }

    @Nonnull
    public Set<String> getIssueIds() {
        return issueIds;
    }

    @Nonnull
    public Set<String> getNodeNames() {
        return nodeNames;
    }

    @Nonnull
    public Set<String> getPluginKeys() {
        return pluginKeys;
    }

    @Nonnull
    public Set<Severity> getSeverities() {
        return severities;
    }

    @Nonnull
    public Optional<Instant> getSince() {
        return ofNullable(since);
    }

    @Nonnull
    public Optional<Instant> getUntil() {
        return ofNullable(until);
    }

    private static <T> Set<T> checkMaxSizeNotExceeded(Set<T> values, String name) {
        if (values.size() > MAX_VALUES_IN_FIELD) {
            throw new IllegalArgumentException("No more than " + MAX_VALUES_IN_FIELD + " " + name +
                    " can be provided (was " + values.size() + ")");
        }
        return values;
    }

    public static class Builder {

        private final ImmutableSet.Builder<String> componentIds;
        private final ImmutableSet.Builder<String> issueIds;
        private final ImmutableSet.Builder<String> nodes;
        private final ImmutableSet.Builder<String> pluginKeys;
        private final ImmutableSet.Builder<Severity> severities;

        private Instant since;
        private Instant until;

        public Builder() {
            componentIds = ImmutableSet.builder();
            issueIds = ImmutableSet.builder();
            nodes = ImmutableSet.builder();
            pluginKeys = ImmutableSet.builder();
            severities = ImmutableSet.builder();
        }

        public Builder(@Nonnull AlertCriteria other) {
            this();

            requireNonNull(other, "other");
            componentIds.addAll(other.componentIds);
            issueIds.addAll(other.issueIds);
            nodes.addAll(other.nodeNames);
            pluginKeys.addAll(other.pluginKeys);
            severities.addAll(other.severities);
            since = other.since;
            until = other.until;
        }

        @Nonnull
        public AlertCriteria build() {
            return new AlertCriteria(this);
        }

        @Nonnull
        public Builder componentIds(String value, String... moreValues) {
            addIf(StringUtils::isNotBlank, TO_UPPER, componentIds, value, moreValues);
            return this;
        }

        @Nonnull
        public Builder componentIds(Iterable<String> values) {
            addIf(StringUtils::isNotBlank, TO_UPPER, componentIds, values);
            return this;
        }

        @Nonnull
        public Builder issueIds(String value, String... moreValues) {
            addIf(StringUtils::isNotBlank, TO_UPPER, issueIds, value, moreValues);
            return this;
        }

        @Nonnull
        public Builder issueIds(Iterable<String> values) {
            addIf(StringUtils::isNotBlank, TO_UPPER, issueIds, values);
            return this;
        }

        @Nonnull
        public Builder nodeNames(String value, String... moreValues) {
            addIf(StringUtils::isNotBlank, nodes, value, moreValues);
            return this;
        }

        @Nonnull
        public Builder nodeNames(Iterable<String> values) {
            addIf(StringUtils::isNotBlank, nodes, values);
            return this;
        }

        @Nonnull
        public Builder pluginKeys(String value, String... moreValues) {
            addIf(StringUtils::isNotBlank, pluginKeys, value, moreValues);
            return this;
        }

        @Nonnull
        public Builder pluginKeys(Iterable<String> values) {
            addIf(StringUtils::isNotBlank, pluginKeys, values);
            return this;
        }

        @Nonnull
        public Builder severities(Severity value, Severity... moreValues) {
            addIf(Objects::nonNull, severities, value, moreValues);
            return this;
        }

        @Nonnull
        public Builder severities(Iterable<Severity> values) {
            addIf(Objects::nonNull, severities, values);
            return this;
        }

        @Nonnull
        public Builder since(Instant value) {
            this.since = value;
            return this;
        }

        @Nonnull
        public Builder until(Instant value) {
            this.until = value;
            return this;
        }

        @SafeVarargs
        private static <T> void addIf(Predicate<T> filter, ImmutableSet.Builder<T> builder, T value, T... moreValues) {
            addIf(filter, Function.identity(), builder, value, moreValues);
        }

        @SafeVarargs
        private static <T> void addIf(Predicate<T> filter, Function<T, T> transform,
                                      ImmutableSet.Builder<T> builder, T value, T... moreValues) {
            if (filter.test(value)) {
                builder.add(transform.apply(value));
            }
            for (T val : moreValues) {
                if (filter.test(val)) {
                    builder.add(transform.apply(val));
                }
            }
        }

        private static <T> void addIf(Predicate<T> filter, ImmutableSet.Builder<T> builder, Iterable<T> values) {
            addIf(filter, Function.identity(), builder, values);
        }

        private static <T> void addIf(Predicate<T> filter, Function<T, T> transform,
                                      ImmutableSet.Builder<T> builder, Iterable<T> values) {
            if (values != null) {
                for (T value : values) {
                    if (filter.test(value)) {
                        builder.add(transform.apply(value));
                    }
                }
            }
        }
    }
}
