package com.atlassian.db.config.password.ciphers.algorithm.paramters;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.File;

public class EncryptionParameters {
    private final String plainTextPassword;
    private final String algorithm;
    private final String algorithmKey;

    //Optionals
    private final String algorithmParametersFilePath;
    private final String keyFilePath;
    private final String outputFilesBasePath;
    private final Boolean saveAlgorithmParametersToSeparateFile;
    private final Boolean saveSealedObjectToSeparateFile;

    public EncryptionParameters(final Builder builder) {
        this.plainTextPassword = builder.plainTextPassword;
        this.algorithm = builder.algorithm;
        this.algorithmKey = builder.algorithmKey;
        this.algorithmParametersFilePath = builder.algorithmParametersFilePath;
        this.keyFilePath = builder.keyFilePath;
        this.outputFilesBasePath = builder.outputFilesBasePath;
        this.saveAlgorithmParametersToSeparateFile = builder.saveAlgorithmParametersToSeparateFile;
        this.saveSealedObjectToSeparateFile = builder.saveSealedObjectToSeparateFile;
    }

    public String getPlainTextPassword() {
        return plainTextPassword;
    }

    public String getAlgorithm() {
        return algorithm;
    }

    public String getAlgorithmKey() {
        return algorithmKey;
    }

    public String getAlgorithmParametersFilePath() {
        return algorithmParametersFilePath;
    }

    public String getKeyFilePath() {
        return keyFilePath;
    }

    public String getOutputFilesBasePath() {
        return outputFilesBasePath;
    }

    public boolean isSaveAlgorithmParametersToSeparateFile() {
        return BooleanUtils.toBooleanDefaultIfNull(saveAlgorithmParametersToSeparateFile, true);
    }

    public boolean isSaveSealedObjectToSeparateFile() {
        return BooleanUtils.toBooleanDefaultIfNull(saveSealedObjectToSeparateFile, true);
    }

    public static class Builder {

        private String plainTextPassword;
        private String algorithm;
        private String algorithmKey;
        private String algorithmParametersFilePath;
        private String keyFilePath;
        private String outputFilesBasePath;
        private Boolean saveAlgorithmParametersToSeparateFile;
        private Boolean saveSealedObjectToSeparateFile;

        public Builder setPlainTextPassword(final String plainTextPassword) {
            this.plainTextPassword = plainTextPassword;
            return this;
        }

        public Builder setAlgorithm(final String algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        public Builder setAlgorithmKey(final String algorithmKey) {
            this.algorithmKey = algorithmKey;
            return this;
        }

        public Builder setAlgorithmParametersFilePath(final String algorithmParametersFilePath) {
            this.algorithmParametersFilePath = algorithmParametersFilePath;
            return this;
        }

        public Builder setKeyFilePath(final String keyFilePath) {
            this.keyFilePath = keyFilePath;
            return this;
        }

        public Builder setOutputFilesBasePath(final String outputFilesBasePath) {
            if (StringUtils.isNotEmpty(outputFilesBasePath) && !outputFilesBasePath.endsWith(File.separator)) {
                throw new IllegalArgumentException("Base path must be ended with " + File.separator);
            }

            this.outputFilesBasePath = outputFilesBasePath;
            return this;
        }

        public Builder setSaveAlgorithmParametersToSeparateFile(final Boolean saveAlgorithmParametersToSeparateFile) {
            this.saveAlgorithmParametersToSeparateFile = saveAlgorithmParametersToSeparateFile;
            return this;
        }

        public Builder setSaveSealedObjectToSeparateFile(final Boolean saveSealedObjectToSeparateFile) {
            this.saveSealedObjectToSeparateFile = saveSealedObjectToSeparateFile;
            return this;
        }

        public EncryptionParameters build() {
            return new EncryptionParameters(this);
        }
    }

    @Override
    public String toString() {
        return "EncryptionParameters{" +
                "plainTextPassword='<SANITIZED>'"+
                ", algorithm='" + algorithm + '\'' +
                ", algorithmKey='" + algorithmKey + '\'' +
                ", algorithmParametersFilePath='" + algorithmParametersFilePath + '\'' +
                ", keyFilePath='" + keyFilePath + '\'' +
                ", outputFilesBasePath='" + outputFilesBasePath + '\'' +
                ", saveAlgorithmParametersToSeparateFile='" + saveAlgorithmParametersToSeparateFile + '\'' +
                ", saveSealedObjectToSeparateFile='" + saveSealedObjectToSeparateFile + '\'' +
                '}';
    }
}
