package com.atlassian.dc.filestore.api;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.ExperimentalSpi;
import org.slf4j.LoggerFactory;

import javax.annotation.WillNotClose;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * A high-level representation of a store of files.
 */
@ExperimentalApi
@ExperimentalSpi
public interface FileStore {
    /**
     * @return The root path of this {@link FileStore}.
     */
    Path root();

    /**
     * @return a {@link Path} within this {@link FileStore} relative to the {@link #root()}.
     */
    default Path path(String... pathComponents) {
        return root().path(pathComponents);
    }

    /**
     * @return The amount of available space for this {@link FileStore}. If the available space cannot be determined or is undefined, an empty value is returned.
     */
    default Optional<DataSize> getAvailableSpace() {
        return Optional.empty();
    }

    /**
     * @return The amount of total space for this {@link FileStore}. If the available space cannot be determined or is undefined, an empty value is returned.
     */
    default Optional<DataSize> getTotalSpace() {
        return Optional.empty();
    }

    /**
     * Represents a Path within a {@link FileStore}.
     */
    interface Path {

        /**
         * @return a {@link Path} relative to the current path.
         */
        Path path(String... pathComponents);

        /**
         * @return true if a file exists at this path, otherwise false
         * @throws IOException if the existence of the file cannot be established
         */
        boolean fileExists() throws IOException;

        /**
         * Determines if a file exists at the current path.
         * If this cannot be determined due to an exception, then it assumes it does not exist.
         *
         * @return true if a file exists at this path, otherwise false
         */
        default boolean tryFileExists() {
            try {
                return fileExists();
            } catch (IOException ex) {
                LoggerFactory.getLogger(getClass()).warn("Failed to determine if file {} exists; assuming it does not", this, ex);
                return false;
            }
        }

        /**
         * @return a {@link Reader} for the file at this path
         */
        Reader fileReader();

        /**
         * @return a {@link Writer} for the file at this path
         */
        Writer fileWriter();

        /**
         * Deletes the file at this path.
         *
         * @throws java.io.FileNotFoundException If no file exists at this path
         * @throws IOException                   if the file could not be deleted
         */
        void deleteFile() throws IOException;

        /**
         * Attempts to delete the file at this path, if it exists.
         *
         * @return true if the file was deleted, false if it does not exist, or it otherwise could not be deleted.
         */
        default boolean tryDeleteFile() {
            try {
                deleteFile();
                return true;
            } catch (FileNotFoundException ex) {
                LoggerFactory.getLogger(getClass()).debug("Cannot delete non-existent file {}", this, ex);
                return false;
            } catch (IOException ex) {
                LoggerFactory.getLogger(getClass()).warn("Failed to delete file {}", this, ex);
                return false;
            }
        }

        /**
         * Moves the file at this path to a new path. If a file already exists at the new path, it will be overwritten.
         * <p>
         * The default implementation of this method first copies the data from the source file to the target,
         * and then deletes the source file, but concrete implementations may provide a more efficient solution.
         *
         * @throws IOException If the move fails
         */
        default void moveFile(Path toFile) throws IOException {
            copyFile(toFile);
            deleteFile();
        }

        /**
         * Copies the file at this path to a new path. If a file already exists at the new path, it will be overwritten.
         * <p>
         * The default implementation of this method streams the data from the source to the target, but concrete
         * impementations may provide a more efficient solution.
         *
         * @throws IOException If the move fails
         */
        default void copyFile(Path toFile) throws IOException {
            toFile.fileWriter().write(outputStream ->
                    fileReader().consume(inputStream -> {
                        final byte[] buffer = new byte[8192];
                        int length;
                        while ((length = inputStream.read(buffer)) > 0) {
                            outputStream.write(buffer, 0, length);
                        }
                    }));
        }

        /**
         * Copies all files from under the current path, to a new root path, maintaining the relayive path structure of all copied files.
         * The default implementation of this method takes a {@link #snapshot()} of the current path, then {@link #unpack(Snapshot)}s that
         * snapshot at the new path. Concrete implementations may perform a more optimised copy.
         *
         * @since 0.7.0
         */
        default void copyFiles(Path toPath) throws IOException {
            try (Snapshot snapshot = snapshot()) {
                toPath.unpack(snapshot);
            }
        }

        /**
         * Determines the exact size of the file at the given path.
         * The default implementation streams and counts the file contents, but implementators should provide
         * a more efficient implementation.
         *
         * @return The file size
         * @throws IOException if the file size cannpt be determined
         */
        default DataSize getFileSize() throws IOException {
            return fileReader().read(inputStream -> {
                int count = 0;
                while (inputStream.read() != -1) {
                    ++count;
                }
                return DataSize.ofBytes(count);
            });
        }

        /**
         * @return the string representation of this path, relative to {@link FileStore#root()}.
         */
        String getPathName();

        /**
         * @return the string representation of the last component of this path, or empty if this path has no components
         */
        Optional<String> getLeafName();

        /**
         * Finds all files that are descendents of the current path.
         *
         * @throws IOException if the descents cannot be determined.
         */
        Stream<? extends Path> getFileDescendents() throws IOException;

        /**
         * Creates a new snapshot of all files under the current path.
         *
         * @since 0.7.0
         */
        default Snapshot snapshot() throws IOException {
            throw new UnsupportedOperationException("Snapshots not supported");
        }

        /**
         * Unpacks the contents of the given snapshot at the current path.
         *
         * @since 0.7.0
         */
        default void unpack(Snapshot snapshot) throws IOException {
            throw new UnsupportedOperationException("Snapshots not supported");
        }
    }

    /**
     * Represents read operations on a file within a {@link FileStore}.
     */
    @FunctionalInterface
    interface Reader {
        /**
         * Opens a new {@link InputStream} for the file.
         * It is the responsibility of the caller to close the stream when it is no longer required.
         *
         * @return the new {@link InputStream}
         * @throws IOException if the file cannot be opened for reading.
         */
        InputStream openInputStream() throws IOException;

        /**
         * Consumes the contents of the file.
         *
         * @param inputStreamConsumer An {@link InputStreamConsumer} which consumes the file contents.
         * @throws IOException if the file cannot be read
         */
        default void consume(final InputStreamConsumer inputStreamConsumer) throws IOException {
            try (InputStream inputStream = openInputStream()) {
                inputStreamConsumer.consume(inputStream);
            }
        }

        /**
         * Consumes the contents of the file and produces a result.
         *
         * @param inputStreamExtractor An {@link InputStreamExtractor} which consumes the file contents and produces the result.
         * @param <T>                  The type of the result.
         * @return The result of the file content extraction.
         * @throws IOException If the file cannpt be read.
         */
        default <T> T read(final InputStreamExtractor<T> inputStreamExtractor) throws IOException {
            try (InputStream inputStream = openInputStream()) {
                return inputStreamExtractor.extract(inputStream);
            }
        }
    }

    /**
     * Represents write operations on a file in a {@link FileStore}.
     */
    @FunctionalInterface
    interface Writer {
        /**
         * Writes the data from the given {@link InputStream} to the file.
         * It is the responsibility of the caller to ensure that the source stream is closed when no longer required.
         *
         * @param source The {@link InputStream} containing the data to be written.
         * @throws IOException If the file cannot be written, or if the source cannot be read.
         */
        void write(@WillNotClose InputStream source) throws IOException;

        /**
         * Writes the data provided by the given {@link InputStreamSupplier} to the file.
         *
         * @param inputStreamSupplier An {@link InputStreamSupplier} which provides an {@link InputStream}.
         * @throws IOException If the file cannot be written to, or if the source cannot be read.
         */
        default void write(final InputStreamSupplier inputStreamSupplier) throws IOException {
            try (InputStream inputStream = inputStreamSupplier.get()) {
                write(inputStream);
            }
        }

        /**
         * Writes the given byte array to the file.
         *
         * @param data The bytes to be written.
         * @throws IOException If the file cannot be writen to.
         */
        default void write(final byte[] data) throws IOException {
            write(new ByteArrayInputStream(data));
        }

        /**
         * Writes to the file using an {@link OutputStreamWriter}.
         * The default implementation will write the contents into a memory buffer, then the buffer will written to the file.
         * Implementing classes may change this behaviour.
         *
         * @param bufferWriter The {@link OutputStreamWriter} which will write the file contents.
         * @throws IOException If the file cannot be written to.
         */
        default void write(final OutputStreamWriter bufferWriter) throws IOException {
            final ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            bufferWriter.writeTo(buffer);
            write(buffer.toByteArray());
        }
    }

    @FunctionalInterface
    interface InputStreamConsumer {
        /**
         * Consumes the given {@link InputStream}.
         * The method is not required the close the stream when finished.
         *
         * @param inputStream The stream to be be consumed.
         * @throws IOException If the stream cannot be read.
         */
        void consume(@WillNotClose InputStream inputStream) throws IOException;
    }

    @FunctionalInterface
    interface InputStreamExtractor<T> {
        /**
         * Consumes the given {@link InputStream}, and extracts a result.
         * The method is not required the close the stream when finished.
         *
         * @param inputStream The stream to be be consumed.
         * @throws IOException If the stream cannot be read.
         */
        T extract(@WillNotClose InputStream inputStream) throws IOException;
    }

    @FunctionalInterface
    interface OutputStreamWriter {
        /**
         * Write to the given {@link OutputStream}.
         * The method is not required the close the stream when finished.
         *
         * @param outputStream The stream to be be written to.
         * @throws IOException If the stream cannot be written to.
         */
        void writeTo(@WillNotClose OutputStream outputStream) throws IOException;
    }

    @FunctionalInterface
    interface InputStreamSupplier {
        /**
         * Provides an {@link InputStream} for reading from.
         * The caller will close the stream when finished.
         *
         * @throws IOException If the stream cannot be opened.
         */
        InputStream get() throws IOException;
    }
}

